/*
********************************************************************************
** SAE J1699-3 Test Source Code
**
**  Copyright (C) 2002 Drew Technologies. http://j1699-3.sourceforge.net/
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
** This source code, when compiled and used with an SAE J2534-compatible pass
** thru device, is intended to run the tests described in the SAE J1699-3
** document in an automated manner.
**
** This computer program is based upon SAE Technical Report J1699,
** which is provided "AS IS"
**
** See j1699.c for details of how to build and run this test.
**
********************************************************************************
** DATE          MODIFICATION
** 04/26/04      Commented out logic to check SID9 INF8.  It is not specified to
**               be cleared via mode $04 in ISO15031-5.
** 05/01/04      Altered logic to flag non-support of INFOTYPE $08 as
**               failure.  OEM must present phase-in plan to CARB if
**               unsupported.
** 05/11/04      Added logic to verify J1699 Spec 11.5 test case #5.17.1
**               Verify ECU support of INFOTYPE $04.
** 05/11/04      Added logic to verify J1699 Spec 11.5 test case #5.17.1
**               Verify ECU support of INFOTYPE $06.
********************************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <windows.h>
#include "j2534.h"
#include "j1699.h"

/* Function Prototypes */
STATUS VerifyUnsupportedPID (void);
STATUS VerifyVINFormat (void);
//STATUS VerifyCALIDFormat (SID9 *pSid9, unsigned long  SidIndex, unsigned long Inf3NumItems);
STATUS VerifyCALIDFormat (unsigned long  EcuIndex, unsigned long  Inf3NumItems);
int    VerifySid9PidSupportData (void);

/*
*******************************************************************************
** VerifyVehicleInformationSupportAndData -
** Function to verify SID9 vehicle info support and data
*******************************************************************************
*/
//*****************************************************************************
//
//
//
//*****************************************************************************
//	DATE		Modification
//	07/16/03	SF#760692:	Mode 9 model year modifier.
//							Eleminated check for gVIN[9] for determination
//							of vehicle model year with respect to INFOTYPE
//							2, 4, or 6 determination.
//*****************************************************************************
STATUS VerifyVehicleInformationSupportAndData (void)
{
    unsigned long  EcuIndex;
    unsigned long  IdIndex;
	unsigned long  Inf2NumResponses = 0;
    unsigned long  Inf3NumItems[OBD_MAX_ECUS] = {0};
    unsigned long  Inf5NumItems[OBD_MAX_ECUS] = {0};
    unsigned long  Inf7NumItems[OBD_MAX_ECUS] = {0};
    unsigned long  NumCalIds = 0;
    SID_REQ        SidReq;
    SID9          *pSid9;
    unsigned long  SidIndex;
    unsigned long  Sid9Limit;
    unsigned char  fInf1Responded = FALSE;
    unsigned char  fInf2Responded = FALSE;
    unsigned char  fInf3Responded = FALSE;
    unsigned char  fInf4Responded = FALSE;
    unsigned char  fInf5Responded = FALSE;
    unsigned char  fInf6Responded = FALSE;
    unsigned char  fInf7Responded = FALSE;
    unsigned char  fInf8Responded = FALSE;
    PASSTHRU_MSG   RxMsg;
    unsigned long  NumMsgs;
    unsigned long  NumCVN;
    unsigned long  StartTimeMsecs;
    unsigned long  SupportCount;
    int            ModelYear;


	SID9	CALIDCount[OBD_MAX_ECUS]; /* J1699 Rev 11.5 TC# 5.17.7 call for each ECU to report */
	SID9	CVNCount[OBD_MAX_ECUS];	  /* J1699 Rev 11.5 TC# 5.17.9 call for each ECU to report */

	memset (CALIDCount, 0x00, sizeof(SID9) * OBD_MAX_ECUS);
	memset (CVNCount,   0x00, sizeof(SID9) * OBD_MAX_ECUS);

    ModelYear = atoi(gUserModelYear);


    /* Request SID 9 support data */
	if (RequestSID9SupportData () != PASS)
	{
		return FAIL;
	}

    /*
    ** Find a INF that is not supported by any ECU and request it to see if causes
    ** vehicle to drop out of diagnostic mode.
    */
	if (VerifyUnsupportedPID () != PASS)
	{
		return FAIL;
	}

    /*
    ** Determine the number of ECUs that support INF2 (VIN)
    */
    SupportCount = 0;
    for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
    {
        if (IsSid9InfSupported (EcuIndex, 0x02) == TRUE)
        {
            if (++SupportCount > 1)
            {
				LogPrint ("FAILURE: SID9 INF2 supported by multiple controllers!\n");
				ERROR_RETURN;
            }
        }
    }

    /* 
    ** For each INF group
    ** Verify that all SID 9 INF data is valid
	** Request user supplied expected OBD ECU responses 
    */
    for (IdIndex = 0x01; IdIndex <= 0x08 /*0x100*/; IdIndex++)
    {
        /* skip INF supported INFs */
        //if (IdIndex == 0x20 || IdIndex == 0x40 || IdIndex == 0x60 || IdIndex == 0x80 ||
        //    IdIndex == 0xA0 || IdIndex == 0xC0 || IdIndex == 0xE0)
        //    continue;

		/* 8/8/04-Per spec V11.7; Mode 9 Odd Vid's 1,3,5,7 should be ignored for ISO15765 */
		if (gOBDList[gOBDListIndex].Protocol == ISO15765)
		{
			if (IdIndex ==  0x01 || IdIndex ==  0x03 ||IdIndex ==  0x05 ||IdIndex ==  0x07)
				continue; /* i.e. skip odd IdIndex for ISO15765. */
		}

        /* If INF is supported by any ECU, request it */
        if (IsSid9InfSupported (-1, IdIndex) == TRUE)
        {
            SidReq.SID    = 9;
            SidReq.NumIds = 1;
            SidReq.Ids[0] = (unsigned char)IdIndex;

            if ( SidRequest( &SidReq, SID_REQ_NORMAL ) != PASS)
            {
                /*
                ** If request for CVN fails on M/Y 2004 or less, ignore it because it
                ** could be due to the extended (up to 30sec) response time that is
                ** not allowed in 2005 and beyond.
                */
                if ( (IdIndex == INF_TYPE_CVN) && (ModelYear < 2005) )
                {
                    StartTimeMsecs = GetTickCount ();
                    while ((GetTickCount() - StartTimeMsecs) < 30000)
                    {
                        NumMsgs = 1;
                        PassThruReadMsgs (gOBDList[gOBDListIndex].ChannelID,
									      &RxMsg,
                                          &NumMsgs,
										  gOBDMaxResponseTimeMsecs);
                        /* If a message was received, process it */
                        if (NumMsgs == 1)
                        {
                            /* Save all read messages in the log file */
                            LogMsg(&RxMsg, LOG_NORMAL_MSG);
                        }
                    }
                    break;
                }

                LogPrint ("FAILURE: Sid9 INF request failed\n");
                ERROR_RETURN;
            }

            for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
            {
                /* If INF is not supported, skip to next ECU */
                if (IsSid9InfSupported (EcuIndex, IdIndex) == FALSE)
                    continue;

                /* Check the data to see if it is valid */
                pSid9 = (SID9 *)&gOBDResponse[EcuIndex].Sid9Inf[0];

                if (gOBDResponse[EcuIndex].Sid9InfSize == 0)
                {
	                LogPrint ("FAILURE: No Sid9 Inf data\n");
					ERROR_RETURN;
                }

                Sid9Limit = (gOBDList[gOBDListIndex].Protocol == ISO15765) ?
                            1 : (gOBDResponse[EcuIndex].Sid9InfSize / sizeof (SID9));

                for (SidIndex = 0; SidIndex < Sid9Limit; SidIndex++)
                {
                    /* Check various INF values for validity */
                    switch(pSid9[SidIndex].INF)
                    {
                        case INF_TYPE_VIN_COUNT:
                        {
                            fInf1Responded = TRUE;

                            if ( (gOBDList[gOBDListIndex].Protocol != ISO15765) &&
                                 (pSid9[SidIndex].NumItems         != 0x05) )
                            {
                                LogPrint ("FAILURE: SID9 INF1 (VIN Count) NumItems = %d (should be 5)\n",
                                pSid9[SidIndex].NumItems);
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case INF_TYPE_VIN:
                        {
							fInf2Responded = TRUE;

							/* Copy the VIN into the global array, J1699 Rev 11.5
							** section 5.17.5
							*/
							if (gOBDList[gOBDListIndex].Protocol == ISO15765)
							{
								memcpy (&gVIN[0], &pSid9[0].Data[0], 17);

								/* J1699 V 11.5 TC#5.17.5 call for check that there are no
								** pad bytes included in message.
								*/
								if (pSid9[0].Data[20] != 0x00)
								{
									LogPrint ("FAILURE: SID9 INF2 VIN format error, must be 17 chars!\n" );
									ERROR_RETURN;
								}
							}
							else
							{
                                if (SidIndex == 0)
                                    gVIN[0] = pSid9[SidIndex].Data[3];
                                else if (SidIndex < 5)
    							    memcpy(&gVIN[SidIndex*4 - 3], &pSid9[SidIndex].Data[0], 4);
							}

							if ( (gOBDList[gOBDListIndex].Protocol == ISO15765) ||
								 (SidIndex == 4) )
							{
								/* 05/11/04 - Logic added to report failure on VIN support
								**            from multiple controllers.
								*/
								if (++Inf2NumResponses != 0x01) /* Count support of INF2 */
								{
									LogPrint ("FAILURE: SID9 INF2 supported by multiple controllers!\n" );
									ERROR_RETURN;
								}

								/* Isolated VIN verification logic */
								if (VerifyVINFormat () != PASS)
								{
									ERROR_RETURN;
								}
							}
                        }
                        break;
                        case INF_TYPE_CALID_COUNT:
                        {
                            fInf3Responded = TRUE;

                            /* Response should be a multiple of four if not ISO15765 */
                            if ( (gOBDList[gOBDListIndex].Protocol != ISO15765) &&
                                 (pSid9[SidIndex].NumItems & 0x03) )
                            {
                                LogPrint ("FAILURE: SID9 INF3 (CALID Count) NumItems = %d "
                                          "(should be a multiple of 4)\n", pSid9[SidIndex].NumItems);
                                ERROR_RETURN;
                            }

                            Inf3NumItems[EcuIndex] = pSid9[SidIndex].NumItems;

							/* Add to number of CALIDs */
						    NumCalIds += Inf3NumItems[EcuIndex];
                        }
                        break;
                        case INF_TYPE_CALID:
                        {
                            fInf4Responded = TRUE;

    					    CALIDCount[EcuIndex].INF = INF_TYPE_CALID;
						    CALIDCount[EcuIndex].NumItems = (unsigned char)Sid9Limit;

						    if (VerifyCALIDFormat (EcuIndex, Inf3NumItems[EcuIndex]) != PASS)
						    {
                                ERROR_RETURN;
						    }

                            SidIndex = Sid9Limit;       /* continue with next ECU */
                        }
                        break;
                        case INF_TYPE_CVN_COUNT:
                        {
                            fInf5Responded = TRUE;
                            Inf5NumItems[EcuIndex] = pSid9[SidIndex].NumItems;
                        }
                        break;
                        case INF_TYPE_CVN:
                        {
                            fInf6Responded = TRUE;

							CVNCount[EcuIndex].INF = INF_TYPE_CVN;
							CVNCount[EcuIndex].NumItems ++;

                            /* Response should match INF5 if not ISO15765 */
                            if ((gOBDList[gOBDListIndex].Protocol != ISO15765) &&
                                (pSid9[Inf5NumItems[EcuIndex] - 1].NumItems != Inf5NumItems[EcuIndex]))
                            {
                                LogPrint ("FAILURE: SID9 INF6 (CVN) NumItems = %d (should match INF5 CVN Count)\n",
                                            pSid9[Inf5NumItems[EcuIndex] - 1].NumItems);
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case INF_TYPE_IPT_COUNT:
						{
                            fInf7Responded = TRUE;

							/* non-ISO15765 should report 0x08 */
							if ( (gOBDList[gOBDListIndex].Protocol != ISO15765) &&
                                 (pSid9[SidIndex].NumItems != 0x08) )
                            {
                                LogPrint ("FAILURE: SID9 INF7 (IPT Count) NumItems = %d (should be 0x08)\n",
                                         pSid9[SidIndex].NumItems);
                                ERROR_RETURN;
                            }

                            Inf7NumItems[EcuIndex] = pSid9[SidIndex].NumItems;
                        }
                        break;
                        case INF_TYPE_IPT:
                        {
                            fInf8Responded = TRUE;

							/* J1699 V11.5 test case 5.17.11 calls for
							** determination is 32 bytes of data returned.
							*/
							if (gOBDList[gOBDListIndex].Protocol == ISO15765)
							{
								/* For CAN, account for only data values transmitted. */
								if (gOBDResponse[EcuIndex].Sid9InfSize - 0x02 != 32)
								{
									LogPrint ("FAILURE: SID9 INF8 (IPT) Data Size Error = %d (Must be 32 bytes!)\n",
									          (gOBDResponse[EcuIndex].Sid9InfSize - 0x02));
									ERROR_RETURN;
								}

                                /* SID9 INF8 (NODI) must equal $10 */
                                if (gOBDResponse[EcuIndex].Sid9Inf[1] != 0x10)
                                {
                                    LogPrint ("FAILURE: SID9 INF8 (NODI) not equal to $10\n");
                                    ERROR_RETURN;
                                }
							}
							else
							{
								/* For non-CAN, calculate the number of responses.  Expected is
								** 0x08 records.  Each response from controller held in
								** data structure SID9.
								*/
								if (gOBDResponse[EcuIndex].Sid9InfSize / sizeof(SID9) != 0x08)
								{
									LogPrint ("FAILURE: SID9 INF8 (IPT) Data Size Error, Must be 32 bytes!\n");
									ERROR_RETURN;
								}
							}

                            /* Response should match INF7 if not ISO15765 */
                            if ((gOBDList[gOBDListIndex].Protocol != ISO15765) &&
                                (pSid9[Inf7NumItems[EcuIndex] - 1].NumItems != Inf7NumItems[EcuIndex]))
                            {
                                LogPrint ("FAILURE: SID9 INF8 (IPT) NumItems = %d (should match INF7 IPT Count)\n",
                                         pSid9[Inf7NumItems[EcuIndex] - 1].NumItems);
                                ERROR_RETURN;
                            }
                        }
                        break;
						default:
						{
							/* Non-OBD INF type */
						}
						break;
                    }
                }
            }
        }
    }

    /* Verify ECU support for INFOTYPE $04 & $06 */
	/* Mode 9 Prompt 2 must be at least equal to or less than the number of CVNs. */
	NumCVN = 0;	/* Init and use for CVN ECU count */

    for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
    {
		/* Verify ECU supports for INFOTYPE $04 */
        if (IsSid9InfSupported (EcuIndex, 4) == FALSE)
		{
            LogPrint ("FAILURE: SID9 INF type 4 support required for ECU\n");
            ERROR_RETURN;
		}

		/* J1699 Rev 11.5 TC# 5.17.7 call for each ECU to report */
		if (CALIDCount[EcuIndex].NumItems == 0x00)
		{
            LogPrint ("FAILURE: SID9 INF type 4, ECU did not report\n");
            ERROR_RETURN;
		}

		/* All msgs must be accounted for as specified by CALIDCount */
		NumCalIds -= CALIDCount[EcuIndex].NumItems;

		/* If ECU supports INFOTYPE $06 */
        if (IsSid9InfSupported (EcuIndex, 6) == TRUE)
		{
			/* Increment CVN count and evaluate later...*/
			NumCVN++;

			if (CVNCount[EcuIndex].NumItems == 0x00)
			{
			    LogPrint ("FAILURE: SID9 INF type 6, ECU did not report\n");
		        ERROR_RETURN;
			}
		}
	}

	/* Now evaluate 'Mode 9 operator prompt 2 must be at least
	** equal to or less than the number of CVN's.
	*/
	if (NumCVN < gUserNumEcusReprgm)
	{
		LogPrint ("FAILURE: SID9 INF type 6, Prompt 2 must be at least equal to or less than the number of CVNs\n");
		ERROR_RETURN;
	}

	/* reordered test to allow execution prior to other data checks for INF support */
    /* Try group support if ISO15765 */
    if (gOBDList[gOBDListIndex].Protocol == ISO15765)
    {
        if (VerifyGroupVehicleInformationSupport() == FAIL)
		{
			ERROR_RETURN;
		}
    }

	if (fInf2Responded == TRUE)
    {
        if ((fInf4Responded == FALSE) || (fInf6Responded == FALSE))
        {
            LogPrint ("FAILURE: SID9 INF types 2,4 and 6 required for M/Y 2005 and newer failure\n");
            ERROR_RETURN;
        }
    }
	else
	{
		LogPrint ("FAILURE: SID9 INF type 2 required for M/Y 2005 and newer failure\n");
		ERROR_RETURN;
	}

    /* If M/Y 2007 or greater, make sure INFO type 8 is also supported */
	/* Per Draft 11.4, if not supporte then flag in error.  OEM will present
	 * Infotype phase in plane.
	 */
    if ((fInf8Responded == FALSE) && (atoi(gUserModelYear) >= 2005))
    {
        LogPrint ("FAILURE: SID9 INF type 8 required for M/Y 2007 and newer failure\n");
        ERROR_RETURN;
    }

    /* Verify there are at least as many CALIDs as OBD ECUs */
    if ( (fInf3Responded == TRUE) && (gOBDList[gOBDListIndex].Protocol != ISO15765) )
    {
        if (NumCalIds != 0)
        {
            LogPrint ("FAILURE: SID9 INF3 (CALID Count should be at least the number of OBD ECUs)\n");
            ERROR_RETURN;
        }
    }

    return PASS;
}

/******************************************************************************
**
**	Function:	RequestSID9SupportData
**
**	Purpose:	Purpose of this function is to Request SID 9 support data
**
*******************************************************************************
**
**	DATE		MODIFICATION
**	10/22/03	Isolated Request SID 9 support data
**	05/11/04	Added logic, per J1699 ver 11.5 TC 5.17.3, request next
**              unsupported INFOTYPE-support INFOTYPE and verify ECU did
**              not drop out.
**  06/16/04    Added logic to account for upper $E0 limit and validate
**              support.
**  07/15/04    Correct logic error associated with the evaluation of $E0
**              support.
**
*******************************************************************************
*/
STATUS RequestSID9SupportData (void)
{
	unsigned long EcuIndex;
    unsigned long IdIndex;
	unsigned long ulInfSupport;  /* used to determine $E0 support indication */

	SID_REQ SidReq;

	for (IdIndex = 0x00; IdIndex < 0x100; IdIndex += 0x20)
    {
        SidReq.SID = 9;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = (unsigned char)IdIndex;
        if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
		{
            /* There must be a response to PID 0x00 */
            if (IdIndex == 0x00)
            {
                LogPrint ("FAILURE: Sid9 support request failed\n");
                return FAIL;
            }
        }

        /* Check if we need to request the next group */
        for (EcuIndex = 0; EcuIndex < gUserNumEcusReprgm; EcuIndex++)
        {
            if (gOBDResponse[EcuIndex].Sid9InfSupport[IdIndex >> 5].IDBits[3] & 0x01)
            {
                break;
            }
        }
        if (EcuIndex >= gUserNumEcusReprgm)
        {
            break;
        }
    }

  	/* Flag error if ECU indicates no support */
	if (VerifySid9PidSupportData() == FAIL)
	{
		ERROR_RETURN;
	}

	/* Enhance logic to verify support information if request is at upper limit of $E0 */
	if ( IdIndex == 0xE0 )
	{
		/* Init variable to no-support */
		ulInfSupport = 0;

		/* For each ECU */
		for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
		{
			/* If MID is supported, keep looking */
			if ( ( gOBDResponse[EcuIndex].Sid9InfSupport[IdIndex >> 5].IDBits[0]		||
				   gOBDResponse[EcuIndex].Sid9InfSupport[IdIndex >> 5].IDBits[1]	    ||
				   gOBDResponse[EcuIndex].Sid9InfSupport[IdIndex >> 5].IDBits[2]	    ||
                (  gOBDResponse[EcuIndex].Sid9InfSupport[IdIndex >> 5].IDBits[3] & 0xFE ) ) != 0x00)
            {
				/* Flag as support indicated! */
				ulInfSupport = 1;
            }
		}

		/* If no ECU indicated support, flag as error. */
		if ( ulInfSupport == 0 )
		{
			LogPrint ("FAILURE: INF $E0 support failure.  No INF support indicated!\n");
			ERROR_RETURN;
		}
	}
	else
	{
		/*
		** Per J1699 rev 11.5 TC# 5.17.3 - Request request next
		**              unsupported INFOTYPE-support INFOTYPE
		*/
		SidReq.SID = 9;
		SidReq.NumIds = 1;
		SidReq.Ids[0] = (unsigned char)IdIndex += 0x20;

		gIgnoreNoResponse = TRUE;

		if ( SidRequest(&SidReq, SID_REQ_NORMAL) == PASS )
		{
			if (gOBDList[gOBDListIndex].Protocol == ISO15765)
			{
				LogPrint ("FAILURE: TC# 5.17.3 - Unexpected response from ECU!\n");
		        gIgnoreNoResponse = FALSE;
				return FAIL;
			}
		}

		gIgnoreNoResponse = FALSE;
	}

	/*
	** Per J1699 rev 11.5 TC# 5.17.3 - Verify ECU did not drop out.
	*/
	if (VerifyLinkActive() != PASS)
	{
		return FAIL;
	}

	return PASS;
}

//*****************************************************************************
//
//	Function:	VerifyUnsupportedPID
//
//	Purpose:	Purpose of this function is to Request SID 9 unsupported PIDs
//
//*****************************************************************************
//
//	DATE		MODIFICATION
//	10/22/03	Isolated Request SID 9 support data
//
//*****************************************************************************
STATUS VerifyUnsupportedPID (void)
{
    unsigned long IdIndex;

	SID_REQ SidReq;

    /*
    ** Find a INF that is not supported by any ECU and request it to see if causes
    ** vehicle to drop out of diagnostic mode.
    */

    /* For each INF group */
    for (IdIndex = 0x01; IdIndex < 0x100; IdIndex++)
    {
        /* If not supported by any ECUs, request it */
        if (IsSid9InfSupported (-1, IdIndex) == FALSE)
        {
            gIgnoreNoResponse = TRUE;
            SidReq.SID = 9;
            SidReq.NumIds = 1;
            SidReq.Ids[0] = (unsigned char)IdIndex;
            SidRequest(&SidReq, SID_REQ_NORMAL);
            gIgnoreNoResponse = FALSE;

            /* Done */
            break;
        }
	}

	return PASS;
}

/******************************************************************************
**
**	Function:	VerifyVINFormat
**
**	Purpose:	Purpose of this function is to verify the VIN format
**				for correct format.  In the event the format fails
**				defined criteria, an error is returned.
**
*******************************************************************************
**
**	DATE		MODIFICATION
**	10/22/03	Isolated VIN verification logic
**	06/17/04	Update VIN character validation as documented in J1699 version
**              11.6, table 79.
**
******************************************************************************/
STATUS VerifyVINFormat (void)
{
	unsigned long VinIndex;
    int           ModelYear;

	/* Check all VIN characters for validity */
	LogPrint ("VIN = %s\n", gVIN);
	for (VinIndex = 0; VinIndex < 17; VinIndex++)
	{
		if ((gVIN[VinIndex] <  '0') || (gVIN[VinIndex] >  'Z') ||
			(gVIN[VinIndex] == 'I') || (gVIN[VinIndex] == 'O') ||
			(gVIN[VinIndex] == 'Q') || (gVIN[VinIndex] == ':') ||
			(gVIN[VinIndex] == ';') || (gVIN[VinIndex] == '<') ||
			(gVIN[VinIndex] == '>') || (gVIN[VinIndex] == '=') ||
			(gVIN[VinIndex] == '?') || (gVIN[VinIndex] == '@'))
		{
			break;
		}
	}

	if (VinIndex != 17)
	{
		LogPrint ("FAILURE: Invalid VIN information\n");
		return FAIL;
	}

	/* Check if model year matches what user entered */
    ModelYear = atoi(gUserModelYear);

    /* if skip directly to test 10 & 11, then user is not asked for model year */
    if (ModelYear != 0)
    {
	    if (((gVIN[9] == '1') && (ModelYear != 2001)) ||
		    ((gVIN[9] == '2') && (ModelYear != 2002)) ||
		    ((gVIN[9] == '3') && (ModelYear != 2003)) ||
		    ((gVIN[9] == '4') && (ModelYear != 2004)) ||
		    ((gVIN[9] == '5') && (ModelYear != 2005)) ||
		    ((gVIN[9] == '6') && (ModelYear != 2006)) ||
		    ((gVIN[9] == '7') && (ModelYear != 2007)) ||
		    ((gVIN[9] == '8') && (ModelYear != 2008)) ||
		    ((gVIN[9] == '9') && (ModelYear != 2009)) ||
		    ((gVIN[9] == 'A') && (ModelYear != 2010)) ||
		    ((gVIN[9] == 'B') && (ModelYear != 2011)) ||
		    ((gVIN[9] == 'C') && (ModelYear != 2012)) ||
		    ((gVIN[9] == 'D') && (ModelYear != 2013)) ||
		    ((gVIN[9] == 'E') && (ModelYear != 2014)) ||
		    ((gVIN[9] == 'F') && (ModelYear != 2015)) ||
		    ((gVIN[9] == 'G') && (ModelYear != 2016)) ||
		    ((gVIN[9] == 'H') && (ModelYear != 2017)) ||
		    ((gVIN[9] == 'J') && (ModelYear != 2018)) ||
		    ((gVIN[9] == 'K') && (ModelYear != 2019)) ||
		    ((gVIN[9] == 'L') && (ModelYear != 2020)) ||
		    ((gVIN[9] == 'M') && (ModelYear != 2021)) ||
		    ((gVIN[9] == 'N') && (ModelYear != 2022)) ||
		    ((gVIN[9] == 'P') && (ModelYear != 2023)) ||
		    ((gVIN[9] == 'R') && (ModelYear != 2024)) ||
		    ((gVIN[9] == 'S') && (ModelYear != 2025)))
	    {
		    LogPrint ("FAILURE: VIN year character does not match user entry\n");
		    return FAIL;
	    }
    }

	return PASS;
}

//*****************************************************************************
//
//	Function:	VerifyCALIDFormat
//
//	Purpose:	Purpose of this function is to verify the CALID format
//				for correct format.  In the event the format fails
//				defined criteria, an error is returned.
//
//*****************************************************************************
//
//	DATE		MODIFICATION
//	11/01/03	Isolated CALID verification logic
//
//*****************************************************************************
STATUS VerifyCALIDFormat (unsigned long  EcuIndex, unsigned long  Inf3NumItems)
{
    unsigned long  SidIndex;
    unsigned long  Sid9Limit;
    unsigned long  Inf4NumItems;
    unsigned long  ItemIndex;
    unsigned long  ByteIndex;
    char           buffer[20];
    SID9          *pSid9;

    if (gOBDResponse[EcuIndex].Sid9InfSize == 0)
    {
        LogPrint ("FAILURE: ECU %X: No Sid9 Inf data\n", GetEcuId(EcuIndex));
		return FAIL;
    }

    pSid9 = (SID9 *)&gOBDResponse[EcuIndex].Sid9Inf[0];

    if (gOBDList[gOBDListIndex].Protocol == ISO15765)
    {
        Inf4NumItems = pSid9->NumItems;
        for (ItemIndex = 0; ItemIndex < Inf4NumItems; ItemIndex++)
		{
            memcpy (buffer, &(pSid9->Data[ItemIndex * 16]), 16);

            for (ByteIndex = 0; ByteIndex < 16; ByteIndex++)
			{
                if ( (buffer[ByteIndex] < ' ') || (buffer[ByteIndex] > 'z') )
				{
                    break;
				}
			}

            for (; ByteIndex < 16; ByteIndex++)
			{
				if (buffer[ByteIndex] != 0)
				{
                    LogPrint ("FAILURE: CALID not zero padded on right\n");
                    return FAIL;
				}
			}

            buffer[16] = 0;
            LogPrint ("INFORMATION: ECU %X : CALID: %s\n", GetEcuId(EcuIndex), buffer);
		}
    }
    else
    {
	    if (pSid9[Inf3NumItems - 1].NumItems != Inf3NumItems)
	    {
		    LogPrint ("FAILURE: SID9 INF4 (CALID) NumItems = %d (should match INF3 CALID Count %d)\n",
                      pSid9[Inf3NumItems - 1].NumItems, Inf3NumItems);
            return FAIL;
	    }

        Sid9Limit = gOBDResponse[EcuIndex].Sid9InfSize / sizeof (SID9);

        for (SidIndex = 0; SidIndex < Sid9Limit; SidIndex += 4)
        {
            for (ItemIndex = 0, ByteIndex = 0; ItemIndex < 4; ItemIndex++, pSid9++)
            {
                buffer[ByteIndex++] = pSid9->Data[0];
                buffer[ByteIndex++] = pSid9->Data[1];
                buffer[ByteIndex++] = pSid9->Data[2];
                buffer[ByteIndex++] = pSid9->Data[3];
            }

            for (ByteIndex = 0; ByteIndex < 16; ByteIndex++)
			{
                if ( (buffer[ByteIndex] < ' ') || (buffer[ByteIndex] > 'z') )
				{
                    break;
				}
			}

            for (; ByteIndex < 16; ByteIndex++)
			{
				if (buffer[ByteIndex] != 0)
				{
                    LogPrint ("FAILURE: CALID not zero padded on right\n");
                    return FAIL;
				}
			}

            buffer[ByteIndex] = 0;
            LogPrint ("INFORMATION: ECU %X : CALID: %s\n", GetEcuId (EcuIndex), buffer);
        }
    }

    return PASS;
}

/******************************************************************************
//	Function:	ReadVIN
//
//	Purpose:	Purpose of this function is to read the VIN from the ECUs.
//				If an ECU returns a correctly formatted VIN, return TRUE,
//				otherwise return FAIL.
******************************************************************************/
STATUS ReadVIN (void)
{
    unsigned long  EcuIndex;
	unsigned long  NumResponses;
    SID_REQ        SidReq;
    SID9          *pSid9;
    unsigned long  SidIndex;

    /* Request SID 9 support data */
	if (RequestSID9SupportData() != PASS)
	{
		return FAIL;
	}

    /* INF Type Vin Count for non-ISO15765 only */
	if (gOBDList[gOBDListIndex].Protocol != ISO15765)
	{
        /* If INF is supported by any ECU, request it */
        if (IsSid9InfSupported (-1, INF_TYPE_VIN_COUNT) == TRUE)
        {
            SidReq.SID    = 9;
            SidReq.NumIds = 1;
            SidReq.Ids[0] = INF_TYPE_VIN_COUNT;

            if (SidRequest (&SidReq, SID_REQ_NORMAL) != PASS)
            {
                LogPrint ("FAILURE: SID9 INF1 request failed\n");
                return FAIL;
            }

            NumResponses = 0;

            /* check responses */
            for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
            {
                /* Check the data to see if it is valid */
                pSid9 = (SID9 *)&gOBDResponse[EcuIndex].Sid9Inf[0];

                if (gOBDResponse[EcuIndex].Sid9InfSize != 0)
                {
                    for (SidIndex = 0;
                         SidIndex < (gOBDResponse[EcuIndex].Sid9InfSize / sizeof (SID9));
				         SidIndex++ )
                    {
                        if (pSid9[SidIndex].INF == INF_TYPE_VIN_COUNT)
                        {
                            if (pSid9[SidIndex].NumItems != 0x05)
                            {
                                LogPrint ("FAILURE: SID9 INF1 (VIN Count) NumItems = %d (should be 5)\n", pSid9[SidIndex].NumItems);
                                return FAIL;
                            }
                        }
                    }

                    NumResponses++;
                }
            }

            if (NumResponses > 1)
            {
                LogPrint ("FAILURE: %u ECUs responded to VIN COUNT (SID 9 INF 1)\n", NumResponses);
                return FAIL;
            }
        }
    }

    /* If INF is supported by any ECU, request it */
    if (IsSid9InfSupported (-1, INF_TYPE_VIN) == TRUE)
    {
        SidReq.SID    = 9;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = INF_TYPE_VIN;

        if (SidRequest (&SidReq, SID_REQ_NORMAL) != PASS)
        {
            LogPrint ("FAILURE: SID9 INF2 request failed\n");
            return FAIL;
        }

        NumResponses = 0;

        /* check responses */
        for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
        {
            /* Check the data to see if it is valid */
            pSid9 = (SID9 *)&gOBDResponse[EcuIndex].Sid9Inf[0];

            for (SidIndex = 0;
                 SidIndex < (gOBDResponse[EcuIndex].Sid9InfSize / sizeof (SID9));
				 SidIndex++ )
            {
                if (pSid9[SidIndex].INF == INF_TYPE_VIN)
                {
					/* Copy the VIN into the global array, J1699 Rev 11.5 section 5.17.5 */
					if (gOBDList[gOBDListIndex].Protocol == ISO15765)
					{
						memcpy (gVIN, &pSid9[0].Data[0], 17);

						/* J1699 V 11.5 TC#5.17.5 call for check that there are no
						** pad bytes included in message.
						*/
						if (pSid9[0].Data[20] != 0x00)
						{
							LogPrint ("FAILURE: SID9 INF2 VIN format error, must be 17 chars!\n");
							return FAIL;
						}
					}
					else
					{
                        if (SidIndex == 0)
                        {
						    gVIN[0] = pSid9[SidIndex].Data[3];
                        }
                        else if (SidIndex < 5)
                        {
						    memcpy (&gVIN[SidIndex*4 - 3], &pSid9[SidIndex].Data[0], 4);
                        }
					}

					if ( (gOBDList[gOBDListIndex].Protocol == ISO15765) || (SidIndex == 4) )
					{
						if (++NumResponses != 0x01) /* only 1 ECU allowed to support VIN */
						{
							LogPrint ("FAILURE: SID9 INF2 supported by multiple controllers!\n");
							return FAIL;
						}

						if (VerifyVINFormat () != PASS)
						{
							return FAIL;
						}
					}
                }
            }
        }

        return PASS;
    }

    /* No VIN support */
    return FAIL;
}

//*****************************************************************************
//  Function:	VerifySid9PidSupportData
//
//	Purpose:	Verify each controller supports at a minimum one PID. 
//              Any ECU that responds that does not support at least 
//              one PID is flagged as an error.
//
//*****************************************************************************
int VerifySid9PidSupportData (void)
{
	int				bReturn = PASS;
    int             bEcuResult;
	unsigned long	EcuIndex;
    unsigned long   Index;

	/* For each ECU */
	for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
	{
        bEcuResult = FAIL;
		for (Index = 0; Index < gOBDResponse[EcuIndex].Sid9InfSupportSize; Index++)
		{
			/* If MID is supported, keep looking */
			if ( ( gOBDResponse[EcuIndex].Sid9InfSupport[Index].IDBits[0]		||
				   gOBDResponse[EcuIndex].Sid9InfSupport[Index].IDBits[1]	    ||
			       gOBDResponse[EcuIndex].Sid9InfSupport[Index].IDBits[2]	    ||
                 ( gOBDResponse[EcuIndex].Sid9InfSupport[Index].IDBits[3] & 0xFE ) ) != 0x00)
			{
                bEcuResult = PASS;
                break;
			}
		}

        if ((bEcuResult == FAIL) && (gOBDResponse[EcuIndex].Sid9InfSupportSize > 0))
        {
            LogPrint ("INFORMATION: ECU %X SID 8 invalid PID supported PIDs", GetEcuId(EcuIndex));
		    bReturn = FAIL;
        }
	}

	return bReturn;
}

//*****************************************************************************
//
//	Function:	IsSid9InfSuported
//
//	Purpose:	Determine if SID 9 INF is supported on specific ECU.
//              Need to have called RequestSID9SupportData() previously.
//              If EcuIndex < 0 then check all ECUs.
//
//*****************************************************************************
//
//	DATE		MODIFICATION
//	02/10/05	Created common function for this logic.
//
//*****************************************************************************
unsigned int IsSid9InfSupported (unsigned int EcuIndex, unsigned int InfIndex)
{
    unsigned int index0;
    unsigned int index1;
    unsigned int index2;
    unsigned int mask;

    if (InfIndex == 0)
        return TRUE;            // all modules must support SID 09 INF 00

    InfIndex--;

    index1 =  InfIndex >> 5;
    index2 = (InfIndex >> 3) & 0x03;
    mask   = 0x80 >> (InfIndex & 0x07);

    if ((signed int)EcuIndex < 0)
    {
        for (index0 = 0; index0 < gUserNumEcus; index0++)
        {
            if (gOBDResponse[index0].Sid9InfSupport[index1].IDBits[index2] & mask)
                return TRUE;
        }
    }
    else
    {
        if (gOBDResponse[EcuIndex].Sid9InfSupport[index1].IDBits[index2] & mask)
            return TRUE;
    }

    return FALSE;
}

//*****************************************************************************
//
//	Function:	GetSid9Inf8Data
//
//	Purpose:	Copy from gOBDResponse[EcuIndex].Sid9Inf into common,
//              protocol-independent format.
//
//*****************************************************************************
//
//	DATE		MODIFICATION
//	02/16/05	Created common function for this logic.
//
//*****************************************************************************
STATUS GetSid9Inf8Data (unsigned int EcuIndex, SID9INF8 * pSid9Inf8)
{
    unsigned int SidIndex;
    unsigned short * pData;
    SID9 * pSid9;

    memset (pSid9Inf8, 0, sizeof (SID9INF8));

    if (gOBDResponse[EcuIndex].Sid9InfSize == 0)
    {
		return FAIL;
    }

	if (gOBDList[gOBDListIndex].Protocol == ISO15765)
    {
        if (gOBDResponse[EcuIndex].Sid9InfSize - 0x02 != 32)
        {
			LogPrint ("FAILURE: SID9 INF8 (IPT) Data Size Error = %d (Must be 32 bytes!)\n",
			          (gOBDResponse[EcuIndex].Sid9InfSize - 0x02) );
			return FAIL;
        }

        memcpy (pSid9Inf8, &(gOBDResponse[EcuIndex].Sid9Inf[0]), sizeof (SID9INF8));

        pSid9 = (SID9 *)&gOBDResponse[EcuIndex].Sid9Inf[0];

        pSid9Inf8->INF  = 8;
        pSid9Inf8->NODI = 16;

        pData = &(pSid9Inf8->OBDCOND);

        for (SidIndex=2; SidIndex<34; SidIndex+=2)
        {
            *pData++ = gOBDResponse[EcuIndex].Sid9Inf[SidIndex] * 256 
                     + gOBDResponse[EcuIndex].Sid9Inf[SidIndex+1];
        }
    }
    else
    {
		if (gOBDResponse[EcuIndex].Sid9InfSize / sizeof(SID9) != 0x08)
		{
			LogPrint ("FAILURE: SID9 INF8 (IPT) Data Size Error, Must be 32 bytes!\n" );
			return FAIL;
		}

        pSid9 = (SID9 *)&gOBDResponse[EcuIndex].Sid9Inf[0];

        pSid9Inf8->INF  = 8;
        pSid9Inf8->NODI = 16;

        pData = &(pSid9Inf8->OBDCOND);

        for ( SidIndex = 0; SidIndex < (gOBDResponse[EcuIndex].Sid9InfSize / sizeof(SID9)); SidIndex++ )
        {
            if (pSid9[SidIndex].INF != 8)
                return FAIL;

            *pData++ = pSid9[SidIndex].Data[0] * 256 + pSid9[SidIndex].Data[1];
            *pData++ = pSid9[SidIndex].Data[2] * 256 + pSid9[SidIndex].Data[3];
        }
    }

    // data structure is valid
    pSid9Inf8->Flags = 1;

    return PASS;
}

//*****************************************************************************
//
//	Function:	PrintCALIDs
//
//	Purpose:	print CALIDs to the log file
//
//*****************************************************************************
STATUS PrintCALIDs (void)
{
    unsigned long  EcuIndex;
    SID_REQ        SidReq;
    SID9          *pSid9;

    unsigned long  Inf3NumItems[OBD_MAX_ECUS] = {0};


    if (gOBDList[gOBDListIndex].Protocol != ISO15765)
    {
        if (IsSid9InfSupported (-1, 3) == PASS)
        {
            SidReq.SID    = 9;
            SidReq.NumIds = 1;
            SidReq.Ids[0] = 3;

            if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
            {
                LogPrint ("FAILURE: Sid9 Inf3 request failed\n");
                return FAIL;
            }

            for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
            {
                /* If INF is not supported, skip to next ECU */
                if (IsSid9InfSupported (EcuIndex, 3) == FALSE)
                    continue;

                /* Check the data to see if it is valid */
                pSid9 = (SID9 *)&gOBDResponse[EcuIndex].Sid9Inf[0];

                if (gOBDResponse[EcuIndex].Sid9InfSize == 0)
                {
	                LogPrint ("FAILURE: ECU %X: No Sid9 Inf data\n", GetEcuId(EcuIndex));
				    ERROR_RETURN;
                }

                if (pSid9[0].NumItems & 0x03)
                {
                    LogPrint ("FAILURE: ECU %X: SID9 INF3 (CALID Count) NumItems = %d "
                              "(should be a multiple of 4)\n", 
                              GetEcuId(EcuIndex), pSid9[0].NumItems);
                    ERROR_RETURN;
                }

                Inf3NumItems[EcuIndex] = pSid9[0].NumItems;
            }
        }
    }

    SidReq.SID    = 9;
    SidReq.NumIds = 1;
    SidReq.Ids[0] = 4;

    if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
    {
        LogPrint ("FAILURE: Sid9 Inf4 request failed\n");
        return FAIL;
    }

    for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
    {
        if (VerifyCALIDFormat (EcuIndex, Inf3NumItems[EcuIndex]) != PASS)
            ERROR_RETURN;
    }

    return PASS;
}