/*
********************************************************************************
** SAE J1699-3 Test Source Code
**
**  Copyright (C) 2002 Drew Technologies. http://j1699-3.sourceforge.net/
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
** This source code, when compiled and used with an SAE J2534-compatible pass
** thru device, is intended to run the tests described in the SAE J1699-3
** document in an automated manner.
**
** This computer program is based upon SAE Technical Report J1699,
** which is provided "AS IS"
**
** See j1699.c for details of how to build and run this test.
**
********************************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <windows.h>
#include "j2534.h"
#include "j1699.h"

int          VerifySid6PidSupportData (void);
STATUS       RequestSID6SupportData (void);
unsigned int IsSid6MidSupported (unsigned int EcuIndex, unsigned int MidIndex);

/*
*******************************************************************************
** VerifyMonitorTestSupportAndResults -
** Function to verify SID6 monitor test support and results
**
**  DATE		MODIFICATION
**  07/15/04    Correct logic error associated with the evaluation of $E0
**              support.
*******************************************************************************
*/
STATUS VerifyMonitorTestSupportAndResults(void)
{
    unsigned long  EcuIndex;
    unsigned long  IdIndex;
    unsigned long  SidIndex;
    unsigned short u_tmp;
    signed short   s_tmp;
    long           TestValue;
    long           TestLimitMax;
    long           TestLimitMin;
    unsigned char  fA20BSupported = FALSE;
    unsigned char  fA20CSupported = FALSE;

    SID_REQ        SidReq;
    SID6          *pSid6;

    /* Request SID 6 support data */
    if (RequestSID6SupportData() != PASS)
        return FAIL;

    /* Verify that all SID 6 test data is reset or within limits */
    for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
    {
        /* For each MID group */
        for (IdIndex = 0x01; IdIndex < 0x100; IdIndex++)
        {
            /* skip PID supported PIDs */
            if (IdIndex == 0x20 || IdIndex == 0x40 || IdIndex == 0x60 || IdIndex == 0x80 ||
                IdIndex == 0xA0 || IdIndex == 0xC0 || IdIndex == 0xE0)
                continue;

            /* If MID is supported, request it */
            if (IsSid6MidSupported (EcuIndex, IdIndex) == TRUE)
            {
                SidReq.SID = 6;
                SidReq.NumIds = 1;
                SidReq.Ids[0] = (unsigned char)IdIndex;
                if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
                {
                    LogPrint("FAILURE: Sid6 MID request failed\n");
                    ERROR_RETURN;
                }

                if (gOBDResponse[EcuIndex].Sid6MidSize == 0)
                {
                    LogPrint("FAILURE: No Sid6 MID data\n");
                    ERROR_RETURN;
                }

                /* Check the data that should be reset and / or within limits */
                pSid6 = (SID6 *)&gOBDResponse[EcuIndex].Sid6Mid[0];
                for (SidIndex = 0; SidIndex < (gOBDResponse[EcuIndex].Sid6MidSize / sizeof(SID6)); SidIndex++)
                {
                    /*
                    ** If ISO15765 protocol...
                    */
                    if (gOBDList[gOBDListIndex].Protocol == ISO15765)
                    {
                        if (gOBDIMDriveCycle == FALSE)
                        {
                            /*
							 ** If MID 0x01 thru 0x10, TID 1 thru 4 (constants),
							 ** the value and limits must match
							 */
							if ((pSid6[SidIndex].OBDMID <= 0x10) && (pSid6[SidIndex].SDTID <= 4))
							{
                                if (pSid6[SidIndex].TVHI != pSid6[SidIndex].MINTLHI ||
							        pSid6[SidIndex].TVHI != pSid6[SidIndex].MAXTLHI ||
                                    pSid6[SidIndex].TVLO != pSid6[SidIndex].MINTLLO ||
                                    pSid6[SidIndex].TVLO != pSid6[SidIndex].MAXTLLO)
                                {
                                    LogPrint("WARNING: Sid6 MID test value/limits not same\n");
                                }
                            }
							/* Otherwise, values should be zero after a code clear */
							else if (gOBDEngineRunning == FALSE)
							{
                                if (pSid6[SidIndex].TVHI    != 0 || pSid6[SidIndex].TVLO    != 0 ||
                                    pSid6[SidIndex].MINTLHI != 0 || pSid6[SidIndex].MINTLLO != 0 ||
                                    pSid6[SidIndex].MAXTLHI != 0 || pSid6[SidIndex].MAXTLLO != 0)
                                {
                                    LogPrint("WARNING: Sid6 MID test value/limits not reset\n");
                                }
                            }
                        }

                        /* Check for support of required OBDMID/SDTID values */
                        if (pSid6[SidIndex].OBDMID == 0xA2 && pSid6[SidIndex].SDTID == 0x0B)
                        {
                            fA20BSupported = TRUE;
                        }
                        if (pSid6[SidIndex].OBDMID == 0xA2 && pSid6[SidIndex].SDTID == 0x0C)
                        {
                            fA20CSupported = TRUE;
                        }
                    }

					/* Check the value against the limits */
					if ( pSid6[SidIndex].UASID & 0x80 )
					{
                        /*
                        ** Signed values
                        */
                        s_tmp = (pSid6[SidIndex].TVHI << 8) + pSid6[SidIndex].TVLO;
                        TestValue = s_tmp;

                        s_tmp = (pSid6[SidIndex].MINTLHI << 8) + pSid6[SidIndex].MINTLLO;
                        TestLimitMin = s_tmp;

                        s_tmp = (pSid6[SidIndex].MAXTLHI << 8 ) + pSid6[SidIndex].MAXTLLO;
                        TestLimitMax = s_tmp;

                    }
                    else
                    {
                        /*
                        ** Unsigned values
                        */
                        u_tmp = (pSid6[SidIndex].TVHI << 8) + pSid6[SidIndex].TVLO;
                        TestValue = u_tmp;

                        u_tmp = (pSid6[SidIndex].MINTLHI << 8) + pSid6[SidIndex].MINTLLO;
                        TestLimitMin = u_tmp;

                        u_tmp = (pSid6[SidIndex].MAXTLHI << 8 ) + pSid6[SidIndex].MAXTLLO;
                        TestLimitMax = u_tmp;
                    }

                    if (TestValue < TestLimitMin)
                    {
                        LogPrint("FAILURE: Sid6 MID test value exceeded min\n");
                        ERROR_RETURN;
                    }

                    if (TestValue > TestLimitMax)
                    {
                        LogPrint("FAILURE: Sid6 MID test value exceeded max\n");
                        ERROR_RETURN;
                    }
                }
            }
        }
    }

    /*
    ** If ISO15765 protocol, make sure the required OBDMID/SDTID values are supported
    ** and try group support
    */
    if (gOBDList[gOBDListIndex].Protocol == ISO15765)
    {
        if (fA20BSupported == FALSE)
        {
            LogPrint("FAILURE: Sid6 MID 0xA2 SDTID 0x0B support failed\n");
            ERROR_RETURN;
        }

        if (fA20CSupported == FALSE)
        {
            LogPrint("FAILURE: Sid6 MID 0xA2 SDTID 0x0C support failed\n");
            ERROR_RETURN;
        }

        if (gOBDIMDriveCycle == FALSE)
        {
            /* 4/30/04 - Restructure logic to allow for Link Active test after completion of
		     *           group request
		     */
		    if(VerifyGroupMonitorTestSupport()==FAIL)
		    {
			    return(FAIL);
		    }
        }
    }

    return(PASS);
}

/*
*******************************************************************************
**	Function:	VerifySid6PidSupportData
**
**	Purpose:	Verify each controller supports at a minimum one PID. 
**              Any ECU that responds that does not support at least 
**              one PID is flagged as an error.
**
*******************************************************************************
*/
int VerifySid6PidSupportData (void)
{
	int				bReturn = PASS;
    int             bEcuResult;
	unsigned long	EcuIndex;
    unsigned long   Index;

	/* For each ECU */
	for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
	{
        bEcuResult = FAIL;
		for (Index = 0; Index < gOBDResponse[EcuIndex].Sid6MidSupportSize; Index++)
		{
			/* If MID is supported, keep looking */
			if ( ( gOBDResponse[EcuIndex].Sid6MidSupport[Index].IDBits[0]		||
				   gOBDResponse[EcuIndex].Sid6MidSupport[Index].IDBits[1]	    ||
			       gOBDResponse[EcuIndex].Sid6MidSupport[Index].IDBits[2]	    ||
                 ( gOBDResponse[EcuIndex].Sid6MidSupport[Index].IDBits[3] & 0xFE ) ) != 0x00)
			{
                bEcuResult = PASS;
                break;
			}
		}

        if ((bEcuResult == FAIL) && (gOBDResponse[EcuIndex].Sid6MidSupportSize > 0))
        {
            LogPrint ("INFORMATION: ECU %X SID 6 invalid PID supported PIDs", GetEcuId(EcuIndex));
		    bReturn = FAIL;
        }
	}

	return bReturn;
}

//*****************************************************************************
//
//	Function:	RequestSID6SupportData
//
//	Purpose:	Purpose of this routine is to verify that SID 6 PID 00
//              returns a support record. Continue requesting support
//              PIDs thru the highest supported group.
//
//*****************************************************************************
STATUS RequestSID6SupportData (void)
{
    unsigned long EcuIndex;
    unsigned long IdIndex;
    unsigned long ulMidSupport;  /* used to determine $E0 support indication */

    SID_REQ       SidReq;

    /* Request SID 6 support data */
    for (IdIndex = 0x00; IdIndex < 0x100; IdIndex += 0x20)
    {
        SidReq.SID = 6;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = (unsigned char)IdIndex;
        if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
        {
            /* There must be a response to MID 0x00 */
            if (IdIndex == 0x00)
            {
                LogPrint("FAILURE: Sid6 support request failed\n");
                return FAIL;
            }
        }

        /* Check if we need to request the next group */
        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            if (gOBDResponse[EcuIndex].Sid6MidSupport[IdIndex >> 5].IDBits[3] & 0x01)
            {
                break;
            }
        }
        if (EcuIndex >= gOBDNumEcus)
        {
            break;
        }
    }

	/* Flag error if ECU indicates no support */
	if (VerifySid6PidSupportData() == FAIL)
	{
		ERROR_RETURN;
	}

    if (gOBDIMDriveCycle == FALSE)
    {
	    /* Enhance logic to verify support information if request is at upper limit of $E0 */
	    if (IdIndex == 0xE0)
	    {
			/* Init variable to no-support */
			ulMidSupport = 0;

		    /* For each ECU */
		    for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
		    {
			    /* If MID is supported, keep looking */
			    if ( ( gOBDResponse[EcuIndex].Sid6MidSupport[IdIndex >> 5].IDBits[0]		||
				       gOBDResponse[EcuIndex].Sid6MidSupport[IdIndex >> 5].IDBits[1]	    ||
				       gOBDResponse[EcuIndex].Sid6MidSupport[IdIndex >> 5].IDBits[2]	    ||
                     ( gOBDResponse[EcuIndex].Sid6MidSupport[IdIndex >> 5].IDBits[3] & 0xFE ) ) != 0x00)
                {
				    /* Flag as support indicated! */
			        ulMidSupport = 1;
                }
		    }

		    /* If no ECU indicated support, flag as error. */
		    if ( ulMidSupport == 0 )
		    {
			    LogPrint("FAILURE: MID $E0 support failure.  No OBDMID support indicated!\n");
			    ERROR_RETURN;
		    }
	    }
	    else
	    {
		    /*
		    ** Per J1699 rev 11.5 TC# 5.14.5 - Request request next
		    **              unsupported OBDMID-support OBDMID
		    */
		    SidReq.SID = 6;
		    SidReq.NumIds = 1;
		    SidReq.Ids[0] = (unsigned char)IdIndex += 0x20;

		    gIgnoreNoResponse = TRUE;

		    if ( SidRequest(&SidReq, SID_REQ_NORMAL) == PASS )
		    {
			    if ( gOBDList[gOBDListIndex].Protocol == ISO15765 )
			    {
				    LogPrint("FAILURE: TC# 5.14.5 - Unexpected response from ECU!\n");
		            gIgnoreNoResponse = FALSE;
				    ERROR_RETURN;	/* 8/9/04 - Allow user to continue */
			    }
		    }

		    gIgnoreNoResponse = FALSE;
	    }

	    /*
	    ** Per J1699 rev 11.5 TC# 5.14.5 - Verify ECU did not
	    ** drop out.
	    */
	    if (VerifyLinkActive() != PASS)
	    {
		    return FAIL;
	    }
    }

    return PASS;
}

/*
*******************************************************************************
**	Function:	IsSid6MidSupported
**
**	Purpose:	Determine if SID 6 MID is supported on specific ECU.
**              If EcuIndex < 0 then check all ECUs.
**
*******************************************************************************
*/
unsigned int IsSid6MidSupported (unsigned int EcuIndex, unsigned int MidIndex)
{
    unsigned int index0;
    unsigned int index1;
    unsigned int index2;
    unsigned int mask;

    if (MidIndex == 0)
        return TRUE;            // all modules must support SID 06 MID 00

    MidIndex--;

    index1 =  MidIndex >> 5;
    index2 = (MidIndex >> 3) & 0x03;
    mask   = 0x80 >> (MidIndex & 0x07);

    if ((signed int)EcuIndex < 0)
    {
        for (index0 = 0; index0 < gUserNumEcus; index0++)
        {
            if (gOBDResponse[index0].Sid6MidSupport[index1].IDBits[index2] & mask)
                return TRUE;
        }
    }
    else
    {
        if (gOBDResponse[EcuIndex].Sid6MidSupport[index1].IDBits[index2] & mask)
            return TRUE;
    }

    return FALSE;
}

