/*
********************************************************************************
** SAE J1699-3 Test Source Code
**
**  Copyright (C) 2002 Drew Technologies. http://j1699-3.sourceforge.net/
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
** This source code, when compiled and used with an SAE J2534-compatible pass
** thru device, is intended to run the tests described in the SAE J1699-3
** document in an automated manner.
**
** This computer program is based upon SAE Technical Report J1699,
** which is provided "AS IS"
**
** See j1699.c for details of how to build and run this test.
**
********************************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <windows.h>
#include "j2534.h"
#include "j1699.h"

/* Funtion prototypes */
STATUS RequestSID2SupportData (void);
STATUS RequestSID2Pid2Data (unsigned short *FFSupOnPendFault);
STATUS VerifyFFDTCinDTCList (unsigned short *pFreezeFrameDTC);
STATUS VerifyAllSID2Data (unsigned short *pFreezeFrameDTC);
STATUS VerifySID2PIDNonResponse (void);

int VerifySid2PidSupportData (void);

static unsigned short FreezeFramePendingDTC[OBD_MAX_ECUS] = {0};

/*
*******************************************************************************
** VerifyFreezeFrameSupportAndData - Function to verify SID2 freeze frame support and data
*******************************************************************************
*/
STATUS VerifyFreezeFrameSupportAndData(void)
{
    unsigned short FFSupOnPendFault;
	unsigned short FreezeFrameDTC[OBD_MAX_ECUS];

    /* Initialize data */
	FFSupOnPendFault = FALSE;
  	memset (FreezeFrameDTC, 0, sizeof (FreezeFrameDTC) );

    /* Request SID 2 support data */
	if (RequestSID2SupportData( ) != PASS)
	{
		return FAIL;
	}

	/* Request SID 2 PID 2 data */
	if (RequestSID2Pid2Data (&FFSupOnPendFault) != PASS)
	{
		return FAIL;
	}

    /* Check if a DTC is stored */
	/* 8/8/04; If FF stored on pending fault then verify that it is stored in SID $07 */
    if ( ( gOBDDTCStored    == TRUE ) ||
		 ( FFSupOnPendFault == TRUE ) )		/* If FF stored on pending fault */
    {
        /* Verify the DTC in the freeze frame is in the list of stored DTCs */
		if ( VerifyFFDTCinDTCList (FreezeFrameDTC) != PASS )
		{
			return FAIL;
		}

		/* Verify that all SID 2 PID data is valid */
		if ( VerifyAllSID2Data (FreezeFrameDTC) != PASS )
		{
			return FAIL;
		}
    }
	else
	{
		/* Verify that all SID 2 PID data is valid */
		/* Per J1699 rev 11.6; min support must be verified */
		if ( VerifyAllSID2Data (FreezeFrameDTC) != PASS )
		{
			return FAIL;
		}

		/* Verify for ISO15765 only */
        if (gOBDList[gOBDListIndex].Protocol == ISO15765)
		{
			if ( VerifySID2PIDNonResponse () != PASS )
			{
				return FAIL;
			}
		}
	}

    /* test 6.5.1 - save FF DTC to compare during next test phase (7.5.1) */
    if (TestPhase == eTestPendingDTC)
    {
        unsigned long  EcuIndex;
        
        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            FreezeFramePendingDTC[EcuIndex] = FreezeFrameDTC[EcuIndex];
        }
    }

    /* test 7.5.1 - compare FF DTC to previous test phase (6.5.1) */
    if (TestPhase == eTestConfirmedDTC)
    {
        unsigned long  EcuIndex;
        
        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            if (FreezeFramePendingDTC[EcuIndex] != 0 &&
                FreezeFramePendingDTC[EcuIndex] != FreezeFrameDTC[EcuIndex])
            {
                LogPrint("FAILURE: Confirmed Freeze Frame DTCs don't match Pending FF DTCs\n");
                ERROR_RETURN;
                break;
            }
        }
    }

	/* Try group support if ISO15765 */
	if (gOBDList[gOBDListIndex].Protocol == ISO15765)
	{
		STATUS ret_code = VerifyGroupFreezeFrameSupport();
        if (ret_code != PASS)
            return ret_code;

        /* Tests 6.x - 9.x only */
        if (TestPhase == eTestPendingDTC    || TestPhase == eTestConfirmedDTC        || 
            TestPhase == eTestFaultRepaired || TestPhase == eTestNoFault3DriveCycle)
        {
            return VerifyGroupFreezeFrameResponse ();
        }
	}

    return PASS;
}

//******************************************************************************
//
//	Function:	RequestSID2SupportData
//
//	Purpose:	Purpose of this routine is to verify that SID 2 PID 00
//				returns a support record.  Continue requesting support
//				PIDS thru highest supported group.
//
//******************************************************************************
//	DATE		MODIFICATION
//	10/28/03	Created function based on embedded logic originally
//				contained in the routine 'VerifyFreezeFrameSupportAndData'.
//	07/13/04	Updated to request first unsupported PID support PID.
//******************************************************************************
STATUS RequestSID2SupportData( void )
{
    unsigned long EcuIndex;
    unsigned long IdIndex;
	unsigned long ulPIDSupport;
    SID_REQ SidReq;

	ulPIDSupport = 0;		/* Used to verify that if no PID support; failure may be flagged */

	/* Request SID 2 support data */
    for (IdIndex = 0x00; IdIndex < 0x100; IdIndex += 0x20)
    {
        SidReq.SID    = 2;
        SidReq.NumIds = 2;
        SidReq.Ids[0] = (unsigned char)IdIndex;
        SidReq.Ids[1] = 0; /* Frame #0 */

        if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
        {
            /* There must be a response to PID 0x00 */
            if (IdIndex == 0x00)
            {
                LogPrint("FAILURE: Sid2 support request failed\n");
                return FAIL;
            }
        }

        /* Check if we need to request the next group */
        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
			/* If MID is supported, keep looking */
			if ( ( gOBDResponse[EcuIndex].Sid2PidSupport[IdIndex >> 5].IDBits[0]		||
				   gOBDResponse[EcuIndex].Sid2PidSupport[IdIndex >> 5].IDBits[1]	    ||
				   gOBDResponse[EcuIndex].Sid2PidSupport[IdIndex >> 5].IDBits[2]	    ||
                (  gOBDResponse[EcuIndex].Sid2PidSupport[IdIndex >> 5].IDBits[3] & 0xFE ) ) != 0x00)
            {
                ulPIDSupport = 1;
            }

            if (gOBDResponse[EcuIndex].Sid2PidSupport[IdIndex >> 5].IDBits[3] & 0x01)
            {
                break;
            }

        }
        if (EcuIndex >= gOBDNumEcus)
        {
            break;
        }
    }

	/* Flag error if ECU indicates no support */
	if (VerifySid2PidSupportData() == FAIL)
	{
		ERROR_RETURN;
	}
    
	/* There must be a response to PID 0x00; 5.11.1; spec 11.6 */
	if (ulPIDSupport == 0x00)
	{
		LogPrint("FAILURE: SID 2 no PIDs are supported.\n");
		ERROR_RETURN
	}

	/* Enhance logic to verify support information if request is at upper limit of $E0 */
	if ( IdIndex == 0xE0 )
	{
		/* Init variable to no-support */
		ulPIDSupport = 0;

		/* For each ECU */
		for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
		{
			/* If MID is supported, keep looking */
			if ( ( gOBDResponse[EcuIndex].Sid2PidSupport[IdIndex >> 5].IDBits[0]		||
				   gOBDResponse[EcuIndex].Sid2PidSupport[IdIndex >> 5].IDBits[1]	    ||
				   gOBDResponse[EcuIndex].Sid2PidSupport[IdIndex >> 5].IDBits[2]	    ||
                (  gOBDResponse[EcuIndex].Sid2PidSupport[IdIndex >> 5].IDBits[3] & 0xFE ) ) != 0x00)
            {
                /* Flag as support indicated! */
				ulPIDSupport = 1;
            }
		}

		/* Flag as error if no support indicated in $E0 */
		if (ulPIDSupport == 0x00)
		{
			LogPrint("FAILURE: SID 2 PID $E0 support failure.  No PID support indicated!\n");
			ERROR_RETURN;
		}
	}
	else
	{
		/*
		** Per J1699 rev 11.5 Request request next unsupported OBDMID-support OBDMID
		*/
        SidReq.SID    = 2;
        SidReq.NumIds = 2;
        SidReq.Ids[1] = 0; /* Frame #0 */
		SidReq.Ids[0] = (unsigned char)IdIndex += 0x20;

		gIgnoreNoResponse = TRUE;

		if ( SidRequest(&SidReq, SID_REQ_NORMAL) == PASS )
		{
			/* J1850 & ISO9141 - No response preferred, but positive response allowed */
			if ( gOBDList[gOBDListIndex].Protocol == ISO15765 )
			{
				LogPrint("FAILURE: TC# 5.10.5 - Unexpected response from ECU!\n");
		        gIgnoreNoResponse = FALSE;
				ERROR_RETURN;
			}
		}

		gIgnoreNoResponse = FALSE;
	}

	/*
	** Per J1699 rev 11.5 TC# 5.10.5 - Verify ECU did not drop out.
	*/
	if (VerifyLinkActive() != PASS)
	{
		ERROR_RETURN;
	}

	return PASS;
}

/*******************************************************************************
**
**	Function:	RequestSID2Pid2Data
**
**	Purpose:	Purpose of this routine is to verify that SID 2 PID 02
**				returns stored DTC.  Assuming SID 2 PID 2 was supported
**				by the controller, data should be returned.
**
********************************************************************************
**	DATE		MODIFICATION
**	10/28/03	Created function based on embedded logic originally
**				contained in the routine 'VerifyFreezeFrameSupportAndData'.
**  08/08/04    Update to account for FF support on pending fault.  J1699 
**              version 11.7 specification evaluation criteria outlines logic.
*******************************************************************************/
STATUS RequestSID2Pid2Data( unsigned short *FFSupOnPendFault )
{
    unsigned long EcuIndex;
    SID_REQ SidReq;

    /* Request SID 2 PID 2 data */
    SidReq.SID    = 2;
    SidReq.NumIds = 2;
    SidReq.Ids[0] = 2;
    SidReq.Ids[1] = 0; /* Frame #0 */

    if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
    {
        LogPrint("FAILURE: Sid2Pid2 request failed\n");
        return FAIL;
    }

    /* Check if a DTC is not supposed to be present */
    if ( ( gOBDDTCStored     == FALSE ) &&
		 ( gOBDDTCPending    == FALSE ) &&
		 ( gOBDDTCHistorical == FALSE ) )
    {
        /* Verify that PID 2 Frame 0 has no DTCs stored */
        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            /* If PID 2 supported, there should be data */
            if ( (IsSid2PidSupported(EcuIndex, 2) == TRUE) &&
                 (gOBDResponse[EcuIndex].Sid2PidSize == 0) )
            {
                LogPrint("FAILURE: Insufficient data returned for Sid2Pid2 request\n");
                ERROR_RETURN
            }

            if ( gOBDResponse[EcuIndex].Sid2Pid[1] != 0x00	||
                 gOBDResponse[EcuIndex].Sid2Pid[2] != 0x00	||
                 gOBDResponse[EcuIndex].Sid2Pid[3] != 0x00 )
            {
                LogPrint("FAILURE: Sid2Pid2 Frame%d DTC stored\n", gOBDResponse[EcuIndex].Sid2Pid[1]);
                ERROR_RETURN
            }
        }
    }
	else if ( ( gOBDDTCPending == TRUE  ) &&
		      ( gOBDDTCStored  == FALSE ) )
	{
        /* Determine if FF stored on pending fault */
        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            /* If PID 2 supported, evaluate ECU */
            if (IsSid2PidSupported(EcuIndex, 2) == TRUE)
            {
	            if ( gOBDResponse[EcuIndex].Sid2Pid[1] != 0x00	||
		             gOBDResponse[EcuIndex].Sid2Pid[2] != 0x00	||
			         gOBDResponse[EcuIndex].Sid2Pid[3] != 0x00 )
				{
					LogPrint("INFORMATION: Sid2Pid2 Frame%d DTC stored on pending fault\n", gOBDResponse[EcuIndex].Sid2Pid[1]);
					*FFSupOnPendFault = TRUE;
				}
			}
        }
	}
	return PASS;
}

//******************************************************************************
//
//	Function:	VerifyFFDTCinDTCList
//
//	Purpose:	Purpose of this routine is to verify the DTC in the freeze
//				frame is in the list of stored DTCs.
//
//******************************************************************************
//	DATE		MODIFICATION
//	10/28/03	Created function based on embedded logic originally
//				contained in the routine 'VerifyFreezeFrameSupportAndData'.
//******************************************************************************
STATUS VerifyFFDTCinDTCList( unsigned short *pFreezeFrameDTC )
{
    unsigned long  EcuIndex;
    unsigned long  AllEcuIndex;
    unsigned long  DTCIndex;

    /* Verify the DTC in the freeze frame is in the list of stored DTCs */
    for ( EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++ )
	{
        /* If PID 2 supported, there should be data */
        if ( (IsSid2PidSupported(EcuIndex, 2) == TRUE) &&
		     (gOBDResponse[EcuIndex].Sid2PidSize == 0) )
		{
            LogPrint( "FAILURE: Insufficient data returned for Sid2Pid2 request\n" );
            return FAIL;
        }

        /* Look through all the stored DTCs on all ECUs for a match */
        pFreezeFrameDTC[EcuIndex] = 0;

        /* Skip if no response or no DTCs stored */
        if (gOBDResponse[EcuIndex].Sid2PidSize == 0 ||
            (gOBDResponse[EcuIndex].Sid2Pid[2] == 0 &&
             gOBDResponse[EcuIndex].Sid2Pid[3] == 0) )
            continue;

        for ( AllEcuIndex = 0; AllEcuIndex < gOBDNumEcus; AllEcuIndex++ )
        {
            for ( DTCIndex = 0; DTCIndex < DTCList[AllEcuIndex].Size; DTCIndex += 2 )
            {
                if ( ( gOBDResponse[EcuIndex].Sid2Pid[2] == DTCList[AllEcuIndex].DTC[DTCIndex] )	&&
					 ( gOBDResponse[EcuIndex].Sid2Pid[3] == DTCList[AllEcuIndex].DTC[DTCIndex + 1] ) )
                {
                    /* If we found the matching DTC, save it for later */
                    pFreezeFrameDTC[EcuIndex] =	(gOBDResponse[EcuIndex].Sid2Pid[2] * 256) +
												 gOBDResponse[EcuIndex].Sid2Pid[3];
                    break;
                }
            }
            /* Break outer loop if we found a match */
            if ( DTCIndex < DTCList[AllEcuIndex].Size )
            {
                break;
            }
        }

		if ( AllEcuIndex >= gOBDNumEcus )
        {
            LogPrint("FAILURE: Freeze frame DTC not found in list of stored DTCs on any ECU\n");
            return FAIL;
        }
    }

	return PASS;
}

/*******************************************************************************
**
**	Function:	VerifyAllSID2Data
**
**	Purpose:	Purpose of this routine is to Verify that all SID 2 PID
**				data is valid.  Verification of minimum support list is
**				performed here.
**
********************************************************************************
**	DATE		MODIFICATION
**	10/28/03	Created function based on embedded logic originally
**				contained in the routine 'VerifyFreezeFrameSupportAndData'.
**  07/22/04    Enhanced to allow min PID support check to be performed 
**              w/no stored DTC.
********************************************************************************
*/
STATUS VerifyAllSID2Data( unsigned short *pFreezeFrameDTC )
{
    unsigned long EcuIndex;
    unsigned long IdIndex;

    SID_REQ SidReq;

	unsigned char fPid02Supported = FALSE;
	unsigned char fPid04Supported = FALSE;
	unsigned char fPid05Supported = FALSE;
	unsigned char fPid0CSupported = FALSE;
	unsigned char fPid0DSupported = FALSE;
	unsigned char fPid11Supported = FALSE;

    /* Verify that all SID 2 PID data is valid */
    for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
    {
	    fPid02Supported = FALSE;
	    fPid04Supported = FALSE;
	    fPid05Supported = FALSE;
	    fPid0CSupported = FALSE;
	    fPid0DSupported = FALSE;
	    fPid11Supported = FALSE;

        /* For each PID group */
        for (IdIndex = 0x01; IdIndex < 0x100; IdIndex++)
        {
            /* skip PID Supported PIDs */
            if (IdIndex == 0x00 || IdIndex == 0x20 || IdIndex == 0x40 || IdIndex == 0x60 ||
                IdIndex == 0x80 || IdIndex == 0xA0 || IdIndex == 0xC0 || IdIndex == 0xE0)
                continue;

            /* If PID is supported, request it */
            if (IsSid2PidSupported (EcuIndex, IdIndex) == TRUE)
            {
				/* Set flags if PID supported */
				if (IdIndex == 0x02)
				{
					fPid02Supported = TRUE;
				}
				if (IdIndex == 0x04)
				{
					fPid04Supported = TRUE;
				}
				if (IdIndex == 0x05)
				{
					fPid05Supported = TRUE;
				}
				if (IdIndex == 0x0C)
				{
					fPid0CSupported = TRUE;
				}
				if (IdIndex == 0x0D)
				{
					fPid0DSupported = TRUE;
				}
				if (IdIndex == 0x11)
				{
					fPid11Supported = TRUE;
				}

				/* If DTC is stored then request it to verify that PID is supported. */
				if (pFreezeFrameDTC[EcuIndex] != 0)
				{
					SidReq.SID    = 2;
					SidReq.NumIds = 2;
					SidReq.Ids[0] = (unsigned char)IdIndex;
					SidReq.Ids[1] = 0; /* Frame #0 */
					if ( SidRequest( &SidReq, SID_REQ_NORMAL ) != PASS )
					{
	                    /*
						** There must be a response for ISO15765 protocol when a freeze
						** frame DTC is set
						*/
						if (gOBDList[gOBDListIndex].Protocol == ISO15765)
						{
	                 LogPrint("FAILURE: Sid2 PID request failed\n");
							ERROR_RETURN;
						}
					}

                    if ( ( gOBDResponse[EcuIndex].Sid2PidSize == 0) && 
                         ( pFreezeFrameDTC[EcuIndex] != 0 ) )
                    {
	                    LogPrint("FAILURE: No Sid2 PID data\n");
							  ERROR_RETURN;
                    }
				}
            } // if (IsSid2PidSupported...
        } // end for (IdIndex...

		/* Check for minimum required PIDs */
        if (gOBDResponse[EcuIndex].Sid2PidSupportSize != 0)
        {
		    if (   ( fPid02Supported == FALSE )   ||
			       ( fPid04Supported == FALSE )   ||
			       ( fPid05Supported == FALSE )   ||
			       ( fPid0CSupported == FALSE )   ||
			       ( fPid0DSupported == FALSE )   ||
			     ( ( fPid11Supported == FALSE ) && ( gOBDDieselFlag == FALSE ) ) )
		    {
			    if ( gOBDDieselFlag == FALSE )
			    {
     			    LogPrint("FAILURE: ECU %X: One or more required PIDs (02,04,05,0C,0D and 11) are not supported\n", GetEcuId(EcuIndex));
			    }
			    else
			    {
    			    LogPrint("FAILURE: ECU %X: One or more required PIDs (02,04,05,0C,and 0D) are not supported\n", GetEcuId(EcuIndex));
			    }
			    ERROR_RETURN;
		    }
        }
    } // end for (EcuIndex...

	return PASS;
}

//******************************************************************************
//
//	Function:	VerifySID2PIDNonResponse
//
//	Purpose:	Purpose of this routine is to Verify that all SID 2 PID
//				data will not respond if PID$0002 indicates $0000.
//
//******************************************************************************
//	DATE		MODIFICATION
//	05/05/04	Created function per specification J1699 rev 11.4.
//******************************************************************************
STATUS VerifySID2PIDNonResponse ()
{
    unsigned long IdIndex;
    SID_REQ		  SidReq;

	SidReq.SID    = 2;
    SidReq.NumIds = 2;

    /* For each PID group */
	for (IdIndex = 0x01; IdIndex < 0x100; IdIndex++)
	{
        /* do not request support PID */
        if (IdIndex == 0x00 || IdIndex == 0x20 || IdIndex == 0x40 || IdIndex == 0x60 || 
            IdIndex == 0x80 || IdIndex == 0xA0 || IdIndex == 0xC0 || IdIndex == 0xE0)
            continue;

        /* If PID is supported, request it */
        if (IsSid2PidSupported (-1, IdIndex) == TRUE)
        {
            SidReq.Ids[0] = (unsigned char)IdIndex;
            SidReq.Ids[1] = 0;

            gIgnoreNoResponse = TRUE;
			if ( SidRequest( &SidReq, SID_REQ_NORMAL ) == PASS )
			{
				if (IdIndex != 0x02)
				{
					LogPrint("FAILURE: Sid2 PID $%02X DTC$0000 Response Error.\n", IdIndex);
                    gIgnoreNoResponse = FALSE;
					ERROR_RETURN;
				}
			}
			else
			{
				if (IdIndex == 0x02)
				{
					LogPrint("FAILURE: Sid2 PID $02 DTC$0000 Response Error.\n");
                    gIgnoreNoResponse = FALSE;
					ERROR_RETURN;
				}
			}
		}
	}

    gIgnoreNoResponse = FALSE;

	return PASS;
}


//*****************************************************************************
//  Function:	VerifySid2PidSupportData
//
//	Purpose:	Verify each controller supports at a minimum one PID. 
//              Any ECU that responds that does not support at least 
//              one PID is flagged as an error.
//
//*****************************************************************************
int VerifySid2PidSupportData (void)
{
	int				bReturn = PASS;
    int             bEcuResult;
	unsigned long	EcuIndex;
    unsigned long   Index;

	/* For each ECU */
	for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
	{
        bEcuResult = FAIL;
		for (Index = 0; Index < gOBDResponse[EcuIndex].Sid2PidSupportSize; Index++)
		{
			/* If MID is supported, keep looking */
			if ( ( gOBDResponse[EcuIndex].Sid2PidSupport[Index].IDBits[0]		||
				   gOBDResponse[EcuIndex].Sid2PidSupport[Index].IDBits[1]	    ||
			       gOBDResponse[EcuIndex].Sid2PidSupport[Index].IDBits[2]	    ||
                 ( gOBDResponse[EcuIndex].Sid2PidSupport[Index].IDBits[3] & 0xFE ) ) != 0x00)
			{
                bEcuResult = PASS;
                break;
			}
		}

        if ((bEcuResult == FAIL) && (gOBDResponse[EcuIndex].Sid2PidSupportSize > 0))
        {
            LogPrint ("INFORMATION: ECU %X SID 8 invalid PID supported PIDs", GetEcuId(EcuIndex));
		    bReturn = FAIL;
        }
	}

	return bReturn;
}

//*****************************************************************************
//
//	Function:	IsSid2PidSupported
//
//	Purpose:	Determine if SID 2 PID x is supported on specific ECU.
//              Need to have obtained supported PIDs previously.
//              If EcuIndex < 0 then check all ECUs.
//
//*****************************************************************************
//
//	DATE		MODIFICATION
//	05/24/05	Created common function for this logic.
//
//*****************************************************************************
unsigned int IsSid2PidSupported (unsigned int EcuIndex, unsigned int PidIndex)
{
    int index1;
    int index2;
    int mask;

    if (PidIndex == 0)
        return TRUE;            // all modules must support SID 02 PID 00

    PidIndex--;

    index1 =  PidIndex >> 5;
    index2 = (PidIndex >> 3) & 0x03;
    mask   = 0x80 >> (PidIndex & 0x07);

    if ((signed int)EcuIndex < 0)
    {
        for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
        {
            if (gOBDResponse[EcuIndex].Sid2PidSupport[index1].IDBits[index2] & mask)
                return TRUE;
        }
    }
    else
    {
        if (gOBDResponse[EcuIndex].Sid2PidSupport[index1].IDBits[index2] & mask)
            return TRUE;
    }

    return FALSE;
}
