/*
********************************************************************************
** SAE J1699-3 Test Source Code
**
**  Copyright (C) 2002 Drew Technologies. http://j1699-3.sourceforge.net/
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
** This source code, when compiled and used with an SAE J2534-compatible pass
** thru device, is intended to run the tests described in the SAE J1699-3
** document in an automated manner.
**
** This computer program is based upon SAE Technical Report J1699,
** which is provided "AS IS"
**
** See j1699.c for details of how to build and run this test.
**
********************************************************************************
**    DATE		MODIFICATION
**    06/14/04  Removed Mode $01 PID $01 / PID $41 comparison.  Implement
**              PID $01 check logic defined in specification
********************************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <windows.h>
#include "j2534.h"
#include "j1699.h"

STATUS VerifyM01P01 (SID1 *pSid1, unsigned long SidIndex);
STATUS GetPid4FArray (void);
unsigned char Pid4F[OBD_MAX_ECUS][4];

/*
*******************************************************************************
** VerifyDiagnosticSupportAndData -
** Function to verify SID1 diagnostic support and data
*******************************************************************************
*/
STATUS VerifyDiagnosticSupportAndData(void)
{
    unsigned long EcuIndex;
    unsigned long IdIndex;
    SID_REQ SidReq;
    SID1 *pSid1;
    unsigned long SidIndex;
    long temp_data_long;
    float temp_data_float;
    unsigned int BSIndex;
	unsigned char fPid01Supported = FALSE;
	unsigned char fPid04Supported = FALSE;
	unsigned char fPid05Supported = FALSE;
	unsigned char fPid0CSupported = FALSE;
	unsigned char fPid0DSupported = FALSE;
	unsigned char fPid11Supported = FALSE;
	unsigned char fPid13Supported = FALSE;	/* Added per V11.7 */
	unsigned char fPid1DSupported = FALSE;	/* Added per V11.7 */
	unsigned char fPid4FSupported = FALSE;

	SID1 Sid1Pid1[OBD_MAX_ECUS];			// SF#??????:	Created veriable to capture the
							                // response from M01P01 response.

    /* Read SID 1 PID support PIDs */
    if (RequestSID1SupportData () != PASS)
    {
        ERROR_RETURN;
    }

	/* Per J1699 rev 11.5 TC# 5.10.5 - Verify ECU did not drop out.	*/
	if (VerifyLinkActive() != PASS)
	{
		ERROR_RETURN;
	}

    /* Determine size of PIDs $06, $07, $08, $09 */
    if (DetermineVariablePidSize () != PASS)
    {
        ERROR_RETURN;
    }
    
    if (GetPid4FArray() != PASS)
    {
        ERROR_RETURN;
    }

    /* For each PID group */
    for (IdIndex = 0x01; IdIndex < 0x100; IdIndex++)
    {
        /* skip PID supported PIDs */
        if (IdIndex == 0x20 || IdIndex == 0x40 || IdIndex == 0x60 || IdIndex == 0x80 ||
            IdIndex == 0xA0 || IdIndex == 0xC0 || IdIndex == 0xE0)
            continue;

        if (IsSid1PidSupported (-1, IdIndex) == FALSE)
            continue;

        SidReq.SID = 1;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = (unsigned char)IdIndex;
        if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
        {
            /* There must be a response for ISO15765 protocol */
            if (gOBDList[gOBDListIndex].Protocol == ISO15765)
            {
                LogPrint("FAILURE: Sid1 PID request failed\n");
                ERROR_RETURN;
            }
        }

        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            if (gOBDResponse[EcuIndex].Sid1PidSize != 0)
            {
                break;
            }
        }

        if (EcuIndex >= gOBDNumEcus)
        {
            LogPrint("FAILURE: No Sid1 PID data\n");
            ERROR_RETURN;
        }

        /* Verify that all SID 1 PID data is valid */
        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            /* Set ECU dependent PID flags */
            fPid13Supported = IsSid1PidSupported (EcuIndex, 0x13);
            fPid1DSupported = IsSid1PidSupported (EcuIndex, 0x1D);
            fPid4FSupported = IsSid1PidSupported (EcuIndex, 0x4F);
            
            /* If PID is supported, check it */
            if (IsSid1PidSupported (EcuIndex, IdIndex) == TRUE)
            {
				/* Set flags if PID supported */
				if (IdIndex == 0x01)
				{
					fPid01Supported = TRUE;
				}
				if (IdIndex == 0x04)
				{
					fPid04Supported = TRUE;
				}
				if (IdIndex == 0x05)
				{
					fPid05Supported = TRUE;
				}
				if (IdIndex == 0x0C)
				{
					fPid0CSupported = TRUE;
				}
				if (IdIndex == 0x0D)
				{
					fPid0DSupported = TRUE;
				}
				if (IdIndex == 0x11)
				{
					fPid11Supported = TRUE;
				}

                /* Check the data to see if it is valid */
                pSid1 = (SID1 *)&gOBDResponse[EcuIndex].Sid1Pid[0];
                for (SidIndex = 0; SidIndex < (gOBDResponse[EcuIndex].Sid1PidSize / sizeof(SID1)); SidIndex++)
                {
                    /* Check various PID values for validity based on vehicle state */
                    switch(pSid1[SidIndex].PID)
                    {
                        case 0x01:
                        {
                            // Capture the response from PID $01!
						    // for use with PID $41 verification logic.
						    memcpy( &Sid1Pid1[EcuIndex], &pSid1[SidIndex], sizeof( SID1 ) );

						    if ( VerifyM01P01( pSid1, SidIndex ) != PASS )
						    {
							    ERROR_RETURN;
						    }
                        }
                        break;
                        case 0x02:
                        {
                            if (gOBDEngineWarm == FALSE)
                            {
                                if ((pSid1[SidIndex].Data[0] != 0x00) ||
                                    (pSid1[SidIndex].Data[1] != 0x00))
                                {
                                    LogPrint("FAILURE: Freeze frames available\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x03:
                        {
                            if (((pSid1[SidIndex].Data[0] & 0xE0) != 0x00) ||
                                ((pSid1[SidIndex].Data[1] & 0xE0) != 0x00))
                            {
                                LogPrint("FAILURE: Reserved bits set\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x04:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 100) / 255);
                            LogPrint("INFORMATION: LOAD_PCT = %d%%\n", temp_data_long);
                            if (gOBDEngineRunning == TRUE)
                            {
                                if (temp_data_long > 60)
                                {
                                    /* Load should be 60% or less with engine running */
                                    LogPrint("FAILURE: LOAD_PCT > 60%%\n");
                                    ERROR_RETURN;
                                }
                            }
                            else
                            {
                                if (temp_data_long != 0)
                                {
                                    /* There should be no load with the engine OFF */
                                    LogPrint("FAILURE: LOAD_PCT > 0%%\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x05:
                        {
                            temp_data_long = pSid1[SidIndex].Data[0] - 40;
                            LogPrint("INFORMATION: ECT = %d C\n", temp_data_long);
                            if (gOBDEngineWarm == TRUE)
                            {
                                if ((temp_data_long < 65) || (temp_data_long > 120))
                                {
                                    LogPrint("FAILURE: ECT exceeded normal range\n");
                                    ERROR_RETURN;
                                }
                            }
                            else
                            {
                                if ((temp_data_long < -20) || (temp_data_long > 120))
                                {
                                    LogPrint("FAILURE: ECT exceeded normal range\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x06:
                        {
                            if (gOBDEngineWarm == TRUE)
                            {
                                temp_data_float = (float)(pSid1[SidIndex].Data[0]) * (float)(100.0/128.0) - (float)100.0;
                                LogPrint("INFORMATION: SHRTFT1 = %f %%\n", temp_data_float);
                                if (temp_data_float < -50.0 || temp_data_float > 50.0)
                                {
                                    LogPrint ("FAILURE: SHRTFT1 out of range\n");
                                    ERROR_RETURN;
                                }

                                if (gSid1VariablePidSize == 2)
                                {
                                    temp_data_float = (float)(pSid1[SidIndex].Data[1]) * (float)(100.0/128.0) - (float)100.0;
                                    LogPrint("INFORMATION: SHRTFT1 = %f %%\n", temp_data_float);
                                    if (temp_data_float < -50.0 || temp_data_float > 50.0)
                                    {
                                        LogPrint ("FAILURE: SHRTFT1 out of range\n");
                                        ERROR_RETURN;
                                    }
                                }
                            }
                        }
                        break;
                        case 0x07:
                        {
                            if (gOBDEngineWarm == TRUE)
                            {
                                temp_data_float = (float)(pSid1[SidIndex].Data[0]) * (float)(100.0/128.0) - (float)100.0;
                                LogPrint("INFORMATION: LONGFT1 = %f %%\n", temp_data_float);
                                if (temp_data_float < -50.0 || temp_data_float > 50.0)
                                {
                                    LogPrint ("FAILURE: LONGFT1 out of range\n");
                                    ERROR_RETURN;
                                }

                                if (gSid1VariablePidSize == 2)
                                {
                                    temp_data_float = (float)(pSid1[SidIndex].Data[1]) * (float)(100.0/128.0) - (float)100.0;
                                    LogPrint("INFORMATION: LONGFT1 = %f %%\n", temp_data_float);
                                    if (temp_data_float < -50.0 || temp_data_float > 50.0)
                                    {
                                        LogPrint ("FAILURE: LONGFT1 out of range\n");
                                        ERROR_RETURN;
                                    }
                                }
                            }
                        }
                        break;
                        case 0x08:
                        {
                            if (gOBDEngineWarm == TRUE)
                            {
                                temp_data_float = (float)(pSid1[SidIndex].Data[0]) * (float)(100.0/128.0) - (float)100.0;
                                LogPrint("INFORMATION: SHRTFT1 = %f %%\n", temp_data_float);
                                if (temp_data_float < -50.0 || temp_data_float > 50.0)
                                {
                                    LogPrint ("FAILURE: SHRTFT1 out of range\n");
                                    ERROR_RETURN;
                                }

                                if (gSid1VariablePidSize == 2)
                                {
                                    temp_data_float = (float)(pSid1[SidIndex].Data[1]) * (float)(100.0/128.0) - (float)100.0;
                                    LogPrint("INFORMATION: SHRTFT1 = %f %%\n", temp_data_float);
                                    if (temp_data_float < -50.0 || temp_data_float > 50.0)
                                    {
                                        LogPrint ("FAILURE: SHRTFT1 out of range\n");
                                        ERROR_RETURN;
                                    }
                                }
                            }
                        }
                        break;
                        case 0x09:
                        {
                            if (gOBDEngineWarm == TRUE)
                            {
                                temp_data_float = (float)(pSid1[SidIndex].Data[0]) * (float)(100.0/128.0) - (float)100.0;
                                LogPrint("INFORMATION: LONGFT1 = %f %%\n", temp_data_float);
                                if (temp_data_float < -50.0 || temp_data_float > 50.0)
                                {
                                    LogPrint ("FAILURE: LONGFT1 out of range\n");
                                    ERROR_RETURN;
                                }

                                if (gSid1VariablePidSize == 2)
                                {
                                    temp_data_float = (float)(pSid1[SidIndex].Data[1]) * (float)(100.0/128.0) - (float)100.0;
                                    LogPrint("INFORMATION: LONGFT1 = %f %%\n", temp_data_float);
                                    if (temp_data_float < -50.0 || temp_data_float > 50.0)
                                    {
                                        LogPrint ("FAILURE: LONGFT1 out of range\n");
                                        ERROR_RETURN;
                                    }
                                }
                            }
                        }
                        break;
                        case 0x0A:
                        {
                            temp_data_long = pSid1[SidIndex].Data[0] * 3;
                            LogPrint ("INFORMATION: FRP = %u kPa\n", temp_data_long);
                            if (gOBDEngineRunning == TRUE)
                            {
                                if (temp_data_long == 0)
                                {
                                    LogPrint("FAILURE: FRP must be greater than 0\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x0B:
                        {
                            temp_data_float = (float)(pSid1[SidIndex].Data[0]);
                            if ( fPid4FSupported == TRUE && Pid4F[EcuIndex][3]!=0 )
                            {
                                temp_data_float = temp_data_float * ((float)(Pid4F[EcuIndex][3]*10)/255);
                            }
                            LogPrint ("INFORMATION: MAP = %f kPa\n", temp_data_float);
                            if (gOBDEngineRunning == TRUE)
                            {
                                if (temp_data_float == 0)
                                {
                                    LogPrint("FAILURE: MAP must be greater than 0\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x0C:
                        {
                            temp_data_long = (((pSid1[SidIndex].Data[0] * 256) + pSid1[SidIndex].Data[1]) / 4);

                            LogPrint("INFORMATION: RPM = %d rpm\n", temp_data_long);
                            if (gOBDEngineRunning == TRUE)
                            {
                                /* Per J1699 rev 11.6 - table 41 */
								if ( ( temp_data_long > 2000 ) || ( temp_data_long < 300 ) )
                                {
                                    /* Idle RPM is outside the reasonable range */
                                    LogPrint("FAILURE: RPM exceeded normal range\n");
                                    ERROR_RETURN;
                                }
                            }
                            else
                            {
                                if (temp_data_long != 0)
                                {
                                    /* There should be no RPM with the engine OFF */
                                    LogPrint("FAILURE: RPM > 0 rpm\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x0D:
                        {
                            LogPrint("INFORMATION: VSS = %d km/h\n", pSid1[SidIndex].Data[0]);
                            if (pSid1[SidIndex].Data[0] != 0x00)
                            {
                                /* There should be no vehicle speed when not moving */
                                LogPrint("FAILURE: VSS > 0 km/h\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x0E:
                        {
                            if (gOBDEngineWarm == TRUE)
                            {
                                temp_data_long = (pSid1[SidIndex].Data[0] / 2) - 64;
                                LogPrint("INFORMATION: SPARKADV = %d deg\n", temp_data_long);
                                if ((temp_data_long < -25) || (temp_data_long > 40))
                                {
                                    LogPrint("FAILURE: SPARKADV exceeded normal range\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x0F:
                        {
                            temp_data_long = pSid1[SidIndex].Data[0] - 40;
                            LogPrint("INFORMATION: IAT = %d C\n", temp_data_long);
                            if (gOBDEngineWarm == TRUE)
                            {
                                if ( (temp_data_long < 0) || (temp_data_long > 120) )
                                {
                                    LogPrint("FAILURE: IAT exceeded normal range\n");
                                    ERROR_RETURN;
                                }
                            }
                            else
                            {
                                if ( (temp_data_long < -20) || (temp_data_long > 120) )
                                {
                                    LogPrint("FAILURE: IAT exceeded normal range\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x10:
                        {
                            temp_data_float = (float)((pSid1[SidIndex].Data[0] * 256) +
                            pSid1[SidIndex].Data[1])/(float)100.0;
                            LogPrint("INFORMATION: MAF = %f gm/s\n", temp_data_float);

                            if (gOBDEngineRunning == TRUE)
                            {
                                if (temp_data_float == 0.0)
                                {
                                    /* MAF should not be zero with the engine running */
                                    LogPrint("FAILURE: MAF = 0 gm/s\n");
                                    ERROR_RETURN;
                                }
                            }
                            else
                            {
								/*
								** J1699 version 11.6 table 23, engine off update.
								*/
                                if (temp_data_float > 5.0)
                                {
                                    /* MAF should be zero with the engine OFF */
                                    LogPrint("FAILURE: MAF > 5 gm/s\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x11:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 100) / 255);
                            LogPrint("INFORMATION: TP = %d%%\n", temp_data_long);
                            if (gOBDDieselFlag == 0)
                            {
                                // non-diesel
                                if (temp_data_long > 40)
                                {
                                    /*
                                    ** Throttle position should be
                                    ** 40% or less when not driving
                                    */
                                    LogPrint("FAILURE: TP > 40%\n");
                                    ERROR_RETURN;
                                }
                            }
                            else
                            {
                                // diesel
                                if (temp_data_long > 100)
                                {
                                    /*
                                    ** Throttle position should be
                                    ** 100% or less when not driving
                                    */
                                    LogPrint("FAILURE: TP > 100%\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x1C:
                        {
                            /* Make sure value is in the valid range before the lookup */
                            if (pSid1[SidIndex].Data[0] > 13)
                            {
                                pSid1[SidIndex].Data[0] = 0;
                            }

							/* Updated the following supported IDs per J1699 V11.5 */
                            /* Vehicle should support OBD-II */
                            LogPrint("INFORMATION: OBD_TYPE = %s\n", OBD_TYPE[pSid1[SidIndex].Data[0]]);
                            if (pSid1[SidIndex].Data[0] != 1 &&    /* CARB OBDII */
                            pSid1[SidIndex].Data[0] != 2	&&     /* Federal OBD */
                            pSid1[SidIndex].Data[0] != 3	&&     /* OBDI and OBDII */
                            pSid1[SidIndex].Data[0] != 7	&&     /* EOBD and OBDII */
                            pSid1[SidIndex].Data[0] != 8	&&     /* EOBD and OBD */
                            pSid1[SidIndex].Data[0] != 9	&&     /* EOBD, OBD and OBDII */
                            pSid1[SidIndex].Data[0] != 0x0B &&     /* JOBD and OBD II */
                            pSid1[SidIndex].Data[0] != 0x0D)       /* JOBD, EOBD, and OBD II */
                            {
                                LogPrint("FAILURE: Not an OBD-II vehicle\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
						case 0x13:		/* J1699 V11.5, Added per table 41 */
                        case 0x1D:
                        {
                            unsigned short O2Bit;
                            unsigned short O2Count;

							/* Identify support for PID 0x13 / 0x1D */
							if ( pSid1[SidIndex].PID == 0x13 )
							{
								fPid13Supported = TRUE;
							}
							else
							{
								fPid1DSupported = TRUE;
							}

							/* Evaluate for dual PID / Spark engine support */
							if ( ( fPid13Supported == TRUE )		&&
								 ( fPid1DSupported == TRUE ) )
							{
								LogPrint("FAILURE: PID 13 & 1D both indicated as supported\n");
                                ERROR_RETURN;
							}

                            /* Count the number of O2 sensors */
                            for (O2Bit = 0x01, O2Count = 0; O2Bit != 0x100; O2Bit = O2Bit << 1)
                            {
                                if (pSid1[SidIndex].Data[0] & O2Bit)
                                {
                                    O2Count++;
                                }
                            }
                            LogPrint("INFORMATION: %d O2 Sensors\n", O2Count);

                            /* O2 should all be zero for diesel, otherwise two or more */
                            if (gOBDDieselFlag == TRUE)
                            {
								/* per spec V11.7; diesel may or may not support
								**                 O2 sensors.
                                ** if (O2Count != 0)
                                ** {
                                **    LogPrint("O2S count should be zero for diesel\n");
                                **    ERROR_RETURN;
                                ** }
								*/
                            }
                            else if (O2Count < 2)
                            {
                                LogPrint("FAILURE: O2S < 2\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x1F:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 256) +
                            pSid1[SidIndex].Data[1]);
                            LogPrint("INFORMATION: RUNTM = %d sec\n", temp_data_long);
                            if (gOBDEngineRunning == TRUE)
                            {
                                if (gOBDEngineWarm == TRUE)
                                {
                                    if (temp_data_long <= 300)
                                    {
                                        /* Run time should greater than 300 seconds */
                                        LogPrint("FAILURE: RUNTM = %d sec\n", temp_data_long);
                                        ERROR_RETURN;
                                    }
                                }
                                else
                                {
                                    if (temp_data_long == 0)
                                    {
                                        /* Run time should not be zero if engine is running */
                                        LogPrint("FAILURE: RUNTM = 0 sec\n");
                                        ERROR_RETURN;
                                    }
                                }
                            }
                            else
                            {
                                if (temp_data_long != 0)
                                {
                                    /* Run time should be zero if engine is OFF */
                                    LogPrint("FAILURE: RUNTM > 0 sec\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x21:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 256) +
                            pSid1[SidIndex].Data[1]);
                            LogPrint("INFORMATION: MIL_DIST = %d km\n", temp_data_long);
                            if (temp_data_long != 0x00)
                            {
                                if ( (gOBDEngineWarm == TRUE)     ||            // test 10.12
                                     (gOBDEngineRunning == FALSE) ||            // test 5.6
                                    ((gOBDEngineRunning == TRUE)  &&            // test 5.10
                                     (gOBDEngineWarm == FALSE)    &&
                                     (gOBDResponse[EcuIndex].Sid4Size > 0) &&
                                     (gOBDResponse[EcuIndex].Sid4[0] == 0x44)))
                                {
                                    LogPrint("FAILURE: MIL_DIST > 0 after clearing DTCs\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x22:
                        {
                            temp_data_float = (float)((pSid1[SidIndex].Data[0] * 256) +
                            pSid1[SidIndex].Data[1]) * (float)0.079;
                            LogPrint("INFORMATION: FRP (relative to manifold) = %f kPa\n", temp_data_float);

							/* 06/16/04; Correct logic error.  Previously following check compared
							**           temp_data_long but should compare to actual calculation
							**           stored to temp_data_float.
							*/
                            if (gOBDEngineRunning == TRUE && temp_data_float == 0)
                            {
                                LogPrint("FAILURE: FRP = 0 kPa\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x23:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 256) +
                            pSid1[SidIndex].Data[1]) * 10;
                            LogPrint("INFORMATION: FRP = %ld kPa\n", temp_data_long);
                            if (gOBDEngineRunning == TRUE && temp_data_long == 0)
                            {
                                LogPrint("FAILURE: FRP = 0 kPa\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x24:
                        case 0x25:
                        case 0x26:
                        case 0x27:
                        case 0x28:
                        case 0x29:
                        case 0x2A:
                        case 0x2B:
                        {
                            switch (pSid1[SidIndex].PID)
                            {
                                case 0x24:
                                    BSIndex = 0x11;
                                break;
                                case 0x25:
                                    BSIndex = 0x12;
                                break;
                                case 0x26:
                                    BSIndex = fPid13Supported ? 0x13 : 0x21;
                                break;
                                case 0x27:
                                    BSIndex = fPid13Supported ? 0x14 : 0x22;
                                break;
                                case 0x28:
                                    BSIndex = fPid13Supported ? 0x21 : 0x31;
                                break;
                                case 0x29:
                                    BSIndex = fPid13Supported ? 0x22 : 0x32;
                                break;
                                case 0x2A:
                                    BSIndex = fPid13Supported ? 0x23 : 0x41;
                                case 0x2B:
                                    BSIndex = fPid13Supported ? 0x24 : 0x42;
                            }

                            temp_data_float = (float)(((unsigned long)(pSid1[SidIndex].Data[0] << 8) |
                            pSid1[SidIndex].Data[1]));
                            if ( fPid4FSupported == FALSE || Pid4F[EcuIndex][0] == 0 )
                            {
                                temp_data_float = temp_data_float * (float)0.0000305;
                            }
                            else
                            {
                                temp_data_float = temp_data_float * ((float)(Pid4F[EcuIndex][0])/65535);
                            }

                            LogPrint("INFORMATION: EQ_RAT%x = %f%%\n", BSIndex, temp_data_float);

                            temp_data_float = (float)(((unsigned long)(pSid1[SidIndex].Data[2] << 8) |
                            pSid1[SidIndex].Data[3]));
                            if ( fPid4FSupported == FALSE || Pid4F[EcuIndex][1] == 0 )
                            {
                                temp_data_float = temp_data_float * (float)0.000122;
                            }
                            else
                            {
                                temp_data_float = temp_data_float * ((float)(Pid4F[EcuIndex][1])/65535);
                            }

                            LogPrint("INFORMATION: O2S%x = %fV\n", BSIndex, temp_data_float);
                        }
                        break;
                        case 0x2C:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 100) / 255);
                            LogPrint("INFORMATION: EGR_PCT = %d%%\n", temp_data_long);
                            if (gOBDEngineRunning == TRUE && 
                               (gOBDDieselFlag == TRUE || gOBDHybridFlag == TRUE) )
                            {
                                if (temp_data_long > 60)
                                {
                                    LogPrint("FAILURE: EGR_PCT > 60%\n");
                                    ERROR_RETURN;
                                }
                            }
                            else
                            {
                                if (temp_data_long > 10)
                                {
                                    LogPrint("FAILURE: EGR_PCT > 10%\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x2F:
                        {
                            if (gOBDEngineWarm == TRUE)
                            {
                                temp_data_float = ((float)pSid1[SidIndex].Data[0]) * (float)(100. / 255.);
                                LogPrint("INFORMATION: FLI = %f\n", temp_data_float);
                                if (temp_data_float < 1.0 || temp_data_float > 100.0)
                                {
                                    LogPrint("FAILURE: FLI = %f%%\n", temp_data_float);
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x30:
                        {
                            temp_data_long = pSid1[SidIndex].Data[0];
                            LogPrint("INFORMATION: WARM_UPS = %d\n", temp_data_long);

							/* 06/04/04 - Do not check during Drive cycle */
							/* 08/08/04 - Update per spec 11.7;evaluate 
							**            to 0 or 1 
							*/
                            if (gOBDEngineWarm == TRUE)                     // test 10.12
                            {
                                if (temp_data_long > 4)
                                {
                                    LogPrint("FAILURE: WARM_UPS > 4\n");
                                    ERROR_RETURN;
                                }
                            }
                            else if ( (gOBDEngineRunning == FALSE) ||       // test 5.6
                                     ((gOBDEngineRunning == TRUE)  &&       // test 5.10
                                      (gOBDResponse[EcuIndex].Sid4Size > 0) &&
                                      (gOBDResponse[EcuIndex].Sid4[0] == 0x44)))
                            {
                                if (temp_data_long != 0)
                                {
                                    LogPrint("FAILURE: WARM_UPS > 0\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x31:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 256) +
                            pSid1[SidIndex].Data[1]);
                            LogPrint("INFORMATION: CLR_DIST = %d km\n", temp_data_long);
                            if (gOBDEngineWarm == TRUE)                     // test 10.12
                            {
                                if (temp_data_long >= 50)
                                {
                                    LogPrint("FAILURE: CLR_DIST >= 50km after CARB drive cycle\n");
                                    ERROR_RETURN;
                                }
                            }
                            else if ( (gOBDEngineRunning == FALSE) ||       // test 5.6
                                     ((gOBDEngineRunning == TRUE)  &&       // test 5.10
                                      (gOBDResponse[EcuIndex].Sid4Size > 0) &&
                                      (gOBDResponse[EcuIndex].Sid4[0] == 0x44)))
                            {
                                if (temp_data_long != 0)
                                {
                                    LogPrint("FAILURE: CLR_DIST > 0 km after clearing DTCs\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x33:
                        {
                            LogPrint("INFORMATION: BARO = %d kPa\n", pSid1[SidIndex].Data[0]);
                            if ((pSid1[SidIndex].Data[0] < 71) || (pSid1[SidIndex].Data[0] > 110))
                            {
                                LogPrint("FAILURE: BARO exceeded normal range\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x34:
                        case 0x35:
                        case 0x36:
                        case 0x37:
                        case 0x38:
                        case 0x39:
                        case 0x3A:
                        case 0x3B:
                        {
                            switch (pSid1[SidIndex].PID)
                            {
                                case 0x34:
                                    BSIndex = 0x11;
                                break;
                                case 0x35:
                                    BSIndex = 0x12;
                                break;
                                case 0x36:
                                    BSIndex = fPid13Supported ? 0x13 : 0x21;
                                break;
                                case 0x37:
                                    BSIndex = fPid13Supported ? 0x14 : 0x22;
                                break;
                                case 0x38:
                                    BSIndex = fPid13Supported ? 0x21 : 0x31;
                                break;
                                case 0x39:
                                    BSIndex = fPid13Supported ? 0x22 : 0x32;
                                break;
                                case 0x3A:
                                    BSIndex = fPid13Supported ? 0x23 : 0x41;
                                break;
                                case 0x3B:
                                    BSIndex = fPid13Supported ? 0x24 : 0x42;
                            }

                            temp_data_float = (float)(((unsigned long)(pSid1[SidIndex].Data[0] << 8) |
                            pSid1[SidIndex].Data[1]));
                            if ( fPid4FSupported == FALSE || Pid4F[EcuIndex][0] == 0 )
                            {
                                temp_data_float = temp_data_float * (float)0.0000305;
                            }
                            else
                            {
                                temp_data_float = temp_data_float * ((float)(Pid4F[EcuIndex][0])/65535);
                            }

                            LogPrint("INFORMATION: EQ_RAT%x = %f%%\n", BSIndex, temp_data_float);

                            temp_data_float = (float)(((pSid1[SidIndex].Data[2] << 8) |
                            pSid1[SidIndex].Data[3]));
                            if ( fPid4FSupported == FALSE || Pid4F[EcuIndex][2] == 0 )
                            {
                                temp_data_float = temp_data_float * (float)0.00390625;
                            }
                            else
                            {
                                temp_data_float = temp_data_float * ((float)(Pid4F[EcuIndex][2])/65535);
                            }

                            LogPrint("INFORMATION: O2S%x = %fmA\n", BSIndex, temp_data_float);
                        }
                        break;
                        case 0x41:
                        {
							if ( gOBDEngineRunning == FALSE )
							{
								/* Logic correction required to perform a comparison of PID $01 DATA C vs.
								** PID $41 DATA D.  Previously PID $41 DATA C & D were incorectly compared.
								*/
                                if (gOBDEngineWarm == FALSE)
                                {
									if ( Sid1Pid1[EcuIndex].Data[2] != pSid1[SidIndex].Data[3] )
									{
										LogPrint("WARNING: Monitor support Data C and D compare warning.\n");
									}
                                }
							}
							else
							{
								/* Catalyst monitor */
								if  ( ( ( Sid1Pid1[EcuIndex].Data[2] & 0x01 ) == 0x00 ) &&
										( pSid1[SidIndex].Data[3]    & 0x01 ) == 0x01 )
								{
									LogPrint("FAILURE: Cat Mon: Unsupported monitor must indicate 'complete'\n");
									ERROR_RETURN;
								}

								/* Heated CAT Mon   */
								if  ( ( ( Sid1Pid1[EcuIndex].Data[2] & 0x02 ) == 0x00 ) &&
										( pSid1[SidIndex].Data[3]    & 0x02 ) == 0x02 )
								{
									LogPrint("FAILURE: Heated Cat Mon: Unsupported monitor must indicate 'complete'\n");
									ERROR_RETURN;
								}

								/* EVAP sys         */
								if  ( ( ( Sid1Pid1[EcuIndex].Data[2] & 0x04 ) == 0x00 ) &&
										( pSid1[SidIndex].Data[3]    & 0x04 ) == 0x04 )
								{
									LogPrint("FAILURE: EVAP sys: Unsupported monitor must indicate 'complete'\n");
									ERROR_RETURN;
								}

								/* Secondary AIR    */
								if  ( ( ( Sid1Pid1[EcuIndex].Data[2] & 0x08 ) == 0x00 ) &&
										( pSid1[SidIndex].Data[3]    & 0x08 ) == 0x08 )
								{
									LogPrint("FAILURE: Secondary AIR: Unsupported monitor must indicate 'complete'\n");
									ERROR_RETURN;
								}

								/* A/C Sys mon      */
								if  ( ( ( Sid1Pid1[EcuIndex].Data[2] & 0x10 ) == 0x00 ) &&
										( pSid1[SidIndex].Data[3]    & 0x10 ) == 0x10 )
								{
									LogPrint("FAILURE: AC Sys mon: Unsupported monitor must indicate 'complete'\n");
									ERROR_RETURN;
								}

								/* O2 sens mon      */
								if  ( ( ( Sid1Pid1[EcuIndex].Data[2] & 0x20 ) == 0x00 ) &&
										( pSid1[SidIndex].Data[3]    & 0x20 ) == 0x20 )
								{
									LogPrint("FAILURE: O2 sens mon: Unsupported monitor must indicate 'complete'\n");
									ERROR_RETURN;
								}
	
								/* O2 sens htr Mon  */
								 if  ( ( ( Sid1Pid1[EcuIndex].Data[2] & 0x40 ) == 0x00 ) &&
										( pSid1[SidIndex].Data[3]     & 0x40 ) == 0x40 )
								{
									LogPrint("FAILURE: O2 sens htr Mon: Unsupported monitor must indicate 'complete'\n");
									ERROR_RETURN;
								}
										
								/* EGR sys mon      */
								if  ( ( ( Sid1Pid1[EcuIndex].Data[2] & 0x80 ) == 0x00 ) &&
										( pSid1[SidIndex].Data[3]    & 0x80 ) == 0x80 )
								{
									LogPrint("FAILURE: EGR sys mon: Unsupported monitor must indicate 'complete'\n");
									ERROR_RETURN;
								}
							}
						}
                        break;
                        case 0x43:
                        {
                            temp_data_long = ((((pSid1[SidIndex].Data[0] * 256) +
							pSid1[SidIndex].Data[1]) * 100) / 255);
                            LogPrint("INFORMATION: LOAD_ABS = %d%%\n", temp_data_long);
                            if (gOBDEngineRunning == TRUE)
                            {
                                if (temp_data_long == 0)
                                {
                                    LogPrint("FAILURE: LOAD_ABS = 0%\n");
                                    ERROR_RETURN;
                                }
                            }
                            else
                            {
                                if (temp_data_long != 0)
                                {
                                    LogPrint("FAILURE: LOAD_ABS > 0%\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x44:
                        {
                            if (gOBDEngineWarm == TRUE)
                            {
                                temp_data_float = (float)(((unsigned long)(pSid1[SidIndex].Data[0]) << 8)
                                 | pSid1[SidIndex].Data[1]);
                                if ( fPid4FSupported == FALSE || Pid4F[EcuIndex][0] == 0 )
                                {
                                    temp_data_float = temp_data_float * (float)0.0000305;
                                }
                                else
                                {
                                    temp_data_float = temp_data_float * ((float)(Pid4F[EcuIndex][0])/65535);
                                }
                                LogPrint("INFORMATION: EQ_RAT = %f%%\n", temp_data_float);
                                if ( (temp_data_float < 0.5) || (temp_data_float > 1.5) )
                                {
                                    LogPrint("FAILURE: EQ_RAT exceeded normal range\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x45:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 100) / 255);
                            LogPrint("INFORMATION: TP_R = %d%%\n", temp_data_long);
                            if ((gOBDDieselFlag == FALSE) && (temp_data_long > 50) ||
                                (gOBDDieselFlag == TRUE)  && (temp_data_long > 100))
                            {
                                LogPrint("FAILURE: TP_R exceeded normal range\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x46:
                        {
                            temp_data_long = pSid1[SidIndex].Data[0] - 40;
                            LogPrint("INFORMATION: AAT = %d C\n", temp_data_long);
                            if ((temp_data_long < -20) || (temp_data_long > 85))
                            {
                                LogPrint("FAILURE: AAT exceeded normal range\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x47:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 100) / 255);
                            LogPrint("INFORMATION:TP_B = %d%%\n", temp_data_long);
							/*
							** J1699 V11.5 calls for value to be between 0 - 60 %
							** for non-diesel and 0 - 100 % for diesel
							*/
                            if (gOBDDieselFlag == 0)
							{
                                // non-diesel
								if ( ( temp_data_long > 60 ) || ( temp_data_long < 0 ) )
								{
									LogPrint("FAILURE: TP_B exceeded normal range\n");
									ERROR_RETURN;
								}
							}
							else
							{
                                // diesel
								if ( ( temp_data_long > 100 ) || ( temp_data_long < 0 ) )
								{
									LogPrint("FAILURE: TP_B exceeded normal range\n");
									ERROR_RETURN;
								}
							}
                        }
                        break;
                        case 0x48:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 100) / 255);
                            LogPrint("INFORMATION: TP_C = %d%%\n", temp_data_long);
                            /*
							** J1699 V11.5 calls for value to be between 0 - 60 %
							** for non-diesel and 0 - 100 % for diesel
							*/
                            if (gOBDDieselFlag == 0)
							{
                                // non-diesel
								if ( ( temp_data_long > 60 ) || ( temp_data_long < 0 ) )
								{
									LogPrint("FAILURE: TP_C exceeded normal range\n");
									ERROR_RETURN;
								}
							}
							else
							{
                                // diesel
								if ( ( temp_data_long > 100 ) || ( temp_data_long < 0 ) )
								{
									LogPrint("FAILURE: TP_C exceeded normal range\n");
									ERROR_RETURN;
								}
							}
                        }
                        break;
                        case 0x49:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 100) / 255);
                            LogPrint("INFORMATION: APP_D = %d%%\n", temp_data_long);
                            if (temp_data_long > 40)
                            {
                                LogPrint("FAILURE: APP_D exceeded normal range\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x4A:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 100) / 255);
                            LogPrint("INFORMATION: APP_E = %d%%\n", temp_data_long);
                            if (temp_data_long > 40)
                            {
                                LogPrint("FAILURE: APP_E exceeded normal range\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x4B:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 100) / 255);
                            LogPrint("INFORMATION: APP_F = %d%%\n", temp_data_long);
                            if (temp_data_long > 40)
                            {
                                LogPrint("FAILURE: APP_F exceeded normal range\n");
                                ERROR_RETURN;
                            }
                        }
                        break;
                        case 0x4D:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 256) +
                            pSid1[SidIndex].Data[1]);
                            LogPrint("INFORMATION: MIL_TIME = %d min\n", temp_data_long);

							/* 06/04/04 - Do not check during IM drive cycle. */
                            if (temp_data_long != 0x00)
                            {
                                if ( (gOBDEngineWarm == TRUE)     ||            // test 10.12
                                     (gOBDEngineRunning == FALSE) ||            // test 5.6
                                    ((gOBDEngineRunning == TRUE)  &&            // test 5.10
                                     (gOBDEngineWarm == FALSE)    &&
                                     (gOBDResponse[EcuIndex].Sid4Size > 0) &&
                                     (gOBDResponse[EcuIndex].Sid4[0] == 0x44)))
                                {
                                    LogPrint("FAILURE: MIL_TIME > 0\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
                        case 0x4E:
                        {
                            temp_data_long = ((pSid1[SidIndex].Data[0] * 256) +
                            pSid1[SidIndex].Data[1]);
                            LogPrint("INFORMATION: CLR_TIME = %d min\n", temp_data_long);

							/* 06/04/04 - Do not check during IM drive cycle. */
                            if (gOBDEngineWarm == TRUE)                     // test 10.12
                            {
                                if (temp_data_long > 30)
                                {
                                    LogPrint("FAILURE: CLR_TIME > 30 min\n");
                                    ERROR_RETURN;
                                }
                            }
                            else if ( (gOBDEngineRunning == FALSE) ||       // test 5.6
                                     ((gOBDEngineRunning == TRUE)  &&       // test 5.10
                                      (gOBDResponse[EcuIndex].Sid4Size > 0) &&
                                      (gOBDResponse[EcuIndex].Sid4[0] == 0x44)))
                            {
                                if (temp_data_long != 0)
                                {
                                    LogPrint("FAILURE: CLR_TIME > 0 min\n");
                                    ERROR_RETURN;
                                }
                            }
                        }
                        break;
						default:
						{
							/* Non-OBD PID */
						}
						break;
                    }
                }
            }
        }
    }

	/* J1699 Rev 11.5 calls for verification of min PID support
	** for compression ignition vehicles.  See section section
	** 5.6.1 evaluation criteria for definition.
	*/
	/* Check for minimum required PIDs if not a diesel or hybrid */

	/* 08/08/04: Evaluate min pid support for Hybrid, 
	**           hence remove Hybrid condition. 
	*/
	if (fPid01Supported == FALSE )
	{
       	LogPrint("FAILURE: Required diagnostic PID 01 not supported\n");
	}
	if (fPid04Supported == FALSE )
	{
       	LogPrint("FAILURE: Required diagnostic PID 04 not supported\n");
	}
	if (fPid05Supported == FALSE )
	{
       	LogPrint("FAILURE: Required diagnostic PID 05 not supported\n");
	}
	if (fPid0CSupported == FALSE )
	{
       	LogPrint("FAILURE: Required diagnostic PID 0C not supported\n");
	}
	if (fPid0DSupported == FALSE )
	{
       	LogPrint("FAILURE: Required diagnostic PID 0D not supported\n");
	}
	/* Compression ignition vehicle not required to support PID $11 */
	if ( ( fPid11Supported == FALSE ) && ( gOBDDieselFlag == FALSE ) )
	{
       	LogPrint("FAILURE: Required diagnostic PID 11 not supported\n");
	}

	/*
	** J1699 Rev 11.5 calls for verification of min PID support
	** for compression ignition vehicles.  See section section
	** 5.6.1 evaluation criteria for definition.
	*/

	if (   (fPid01Supported == FALSE) ||
		   (fPid04Supported == FALSE) ||
		   (fPid05Supported == FALSE) ||
		   (fPid0CSupported == FALSE) ||
		   (fPid0DSupported == FALSE) ||
		 ( (fPid11Supported == FALSE) && ( gOBDDieselFlag == FALSE ) ) )
	{
       	ERROR_RETURN;
	}
	else
	{
		LogPrint("INFORMATION: All required PIDs supported!\n");
	}

    /* Try group support if ISO15765 */
    if (gOBDList[gOBDListIndex].Protocol == ISO15765)
    {
        /*return(VerifyGroupDiagnosticSupport());*/
	    /* 4/30/04: Link active test to verify communication remained active for ALL protocols
	     */
		if ( VerifyGroupDiagnosticSupport() == FAIL )
		{
			ERROR_RETURN;
		}
    }

	/* Link active test to verify communication remained active for ALL protocols
	 */
	if (VerifyLinkActive() != PASS)
	{
		ERROR_RETURN;
	}

    return(PASS);
}

/*
********************************************************************************
**	FUNCTION	VerifyM01P01
**
**	Purpose		Isolated function an align to specification SAEJ1699 rev 11.5.
********************************************************************************
**    DATE		MODIFICATION
**	  05/14/04  Created function / aligned test to SAEJ1699 rev 11.5.
********************************************************************************
*/
STATUS VerifyM01P01( SID1 *pSid1, unsigned long SidIndex )
{

   BOOL bTestFailed = FALSE;

	/* Check if the MIL light is ON */
	if (pSid1[SidIndex].Data[0] & 0x80)
	{
		if (gOBDDTCStored == FALSE)
		{
			/* MIL is ON when there should not be a stored DTC */
			LogPrint("FAILURE: MIL status failure\n");
            bTestFailed = TRUE;
		}
	}

	/* Check if any DTC status bits */
	if (pSid1[SidIndex].Data[0] & 0x7F)
	{
		if (gOBDDTCStored == FALSE)
		{
			/*
			** DTC status bit(s) set when there should not
			** be a stored DTC
			*/
			LogPrint("FAILURE: DTC status failure\n");
            bTestFailed = TRUE;
		}
	}

	/* Evaluate Data B, BIT 4 */
	if ( ( ( pSid1[SidIndex].Data[1] & 0x01 ) == 0x00 ) ||    /* Check if ECU does NOT support misfire monitor */
		   ( gOBDDieselFlag == FALSE )                        /* Check for non-compression ignition            */
	   )
	{
		if ( ( pSid1[SidIndex].Data[1] & 0x10 ) != 0x00 )
		{
			LogPrint("FAILURE: Misfire Monitor must be '0' for spark ignition engines or controllers that do not support misfire.\n");
            bTestFailed = TRUE;
		}
	}
	
	/* Evaluate Data B, BIT 4 'or' condition. */
    if ( ( (pSid1[SidIndex].Data[1] & 0x01) == 0x01) &&     /* Check if ECU does NOT support misfire monitor */
		   (gOBDDieselFlag == TRUE) )                       /* Check for non-compression ignition            */
	{
		/* Bit 4 may be 0 or 1 for compression ignition w/Engine running. */
        if (TestPhase == eTestPerformanceCounters)
        {
		    if ((pSid1[SidIndex].Data[1] & 0x10) != 0)
		    {
			    LogPrint("FAILURE: Misfire Monitor must be '0' for compression ignition engines.\n");
			    bTestFailed = TRUE;
		    }
        }
        else
		if ( (gOBDEngineRunning == FALSE) &&
			((pSid1[SidIndex].Data[1] & 0x10) != 0x10) )
		{
			LogPrint("FAILURE: Misfire Monitor must be '1' for compression ignition engines.\n");
			bTestFailed = TRUE;
		}
	}

	/* Evaluate Data B, BIT 5 */
	if ( (pSid1[SidIndex].Data[1] & 0x20 ) == 0x20)
	{
		LogPrint("FAILURE: Fuel system must indicate complete\n");
        bTestFailed = TRUE;
	}

	/* Evaluate Data B, BIT 6 */
	if ( (pSid1[SidIndex].Data[1] & 0x40) == 0x40 )
	{
		LogPrint("FAILURE: CCM must indicate complete\n");
        bTestFailed = TRUE;
	}

	/* Evaluate unsupported monitor status */
	/* If misfire unsupported it must indicate complete */
	if  ( ( ( pSid1[SidIndex].Data[1] & 0x01 ) == 0x00 ) &&
	        ( pSid1[SidIndex].Data[1] & 0x10 ) == 0x10 )
	{
		LogPrint("FAILURE: Misfire test: unsupported monitor must indicate 'complete'\n");
        bTestFailed = TRUE;
	}

	/* If fuel system monitor unsupported it must indicate complete */
	if  ( ( ( pSid1[SidIndex].Data[1] & 0x02 ) == 0x00 ) &&
	        ( pSid1[SidIndex].Data[1] & 0x20 ) == 0x20 )
	{
		LogPrint("FAILURE: Fuel system: unsupported monitor must indicate 'complete'\n");
        bTestFailed = TRUE;
	}

	/* If CCM monitor unsupported it must indicate complete */
	if  ( ( ( pSid1[SidIndex].Data[1] & 0x04 ) == 0x00 ) &&
	        ( pSid1[SidIndex].Data[1] & 0x40 ) == 0x40 )
	{
		LogPrint("FAILURE: CCM unsupported monitor must indicate 'complete'\n");
        bTestFailed = TRUE;
	}

	/* Check if reserved bits 3 and 7 are set */
	if (pSid1[SidIndex].Data[1] & 0x08)
	{
		LogPrint("FAILURE: Reserved I/M readiness status bit 3 set\n");
        bTestFailed = TRUE;
	}

	if (pSid1[SidIndex].Data[1] & 0x80)
	{
		LogPrint("FAILURE: Reserved I/M readiness status bit 7 set\n");
        bTestFailed = TRUE;
	}

    /* If ECU supports M01 P01 then at lease one bit must be set */
	if ( ( ( pSid1[SidIndex].Data[1] & 0x07 ) == 0x00 ) && 
		 ( ( pSid1[SidIndex].Data[2]        ) == 0x00 ) )
	{
		LogPrint("FAILURE: At lease one monitor must be supported by ECU!\n");
        bTestFailed = TRUE;
	}

	if ( gOBDEngineRunning == FALSE )
	{
        /* Don't check O2 sensor heater monitor */
		if ( (pSid1[SidIndex].Data[2] & ~0x40) != (pSid1[SidIndex].Data[3] & ~0x40))
		{
			LogPrint("FAILURE: Supported monitors are indicating 'complete' or unsupported monitors are indicating incomplete.\n");
            bTestFailed = TRUE;
		}
	}
	else
	{
		/* Catalyst monitor */
		if  ( ( ( pSid1[SidIndex].Data[2] & 0x01 ) == 0x00 ) &&
				( pSid1[SidIndex].Data[3] & 0x01 ) == 0x01 )
		{
			LogPrint("FAILURE: Cat Mon: Unsupported monitor must indicate 'complete'\n");
            bTestFailed = TRUE;
		}

		/* Heated CAT Mon   */
		if  ( ( ( pSid1[SidIndex].Data[2] & 0x02 ) == 0x00 ) &&
				( pSid1[SidIndex].Data[3] & 0x02 ) == 0x02 )
		{
			LogPrint("FAILURE: Heated Cat Mon: Unsupported monitor must indicate 'complete'\n");
            bTestFailed = TRUE;
		}

		/* EVAP sys         */
		if  ( ( ( pSid1[SidIndex].Data[2] & 0x04 ) == 0x00 ) &&
				( pSid1[SidIndex].Data[3] & 0x04 ) == 0x04 )
		{
			LogPrint("FAILURE: EVAP sys: Unsupported monitor must indicate 'complete'\n");
            bTestFailed = TRUE;
		}

		/* Secondary AIR    */
		if  ( ( ( pSid1[SidIndex].Data[2] & 0x08 ) == 0x00 ) &&
				( pSid1[SidIndex].Data[3] & 0x08 ) == 0x08 )
		{
			LogPrint("FAILURE: Secondary AIR: Unsupported monitor must indicate 'complete'\n");
            bTestFailed = TRUE;
		}

		/* A/C Sys mon      */
		if  ( ( ( pSid1[SidIndex].Data[2] & 0x10 ) == 0x00 ) &&
				( pSid1[SidIndex].Data[3] & 0x10 ) == 0x10 )
		{
			LogPrint("FAILURE: AC Sys mon: Unsupported monitor must indicate 'complete'\n");
            bTestFailed = TRUE;
		}

		/* O2 sens mon      */
		if  ( ( ( pSid1[SidIndex].Data[2] & 0x20 ) == 0x00 ) &&
				( pSid1[SidIndex].Data[3] & 0x20 ) == 0x20 )
		{
			LogPrint("FAILURE: O2 sens mon: Unsupported monitor must indicate 'complete'\n");
            bTestFailed = TRUE;
		}
		
		/* O2 sens htr Mon  */
		if  ( ( ( pSid1[SidIndex].Data[2] & 0x40 ) == 0x00 ) &&
				( pSid1[SidIndex].Data[3] & 0x40 ) == 0x40 )
		{
			LogPrint("FAILURE: O2 sens htr Mon: Unsupported monitor must indicate 'complete'\n");
            bTestFailed = TRUE;
		}
		

		/* EGR sys mon      */
		if  ( ( ( pSid1[SidIndex].Data[2] & 0x80 ) == 0x00 ) &&
				( pSid1[SidIndex].Data[3] & 0x80 ) == 0x80 )
		{
			LogPrint("FAILURE: EGR sys mon: Unsupported monitor must indicate 'complete'\n");
            bTestFailed = TRUE;
		}

	}

	if (gOBDIMDriveCycle == TRUE)
	{
		if (pSid1[SidIndex].Data[3] != 0)
		{
			LogPrint("FAILURE: Supported monitors not complete after I/M drive cycle\n");
            bTestFailed = TRUE;
		}
	}

   if (bTestFailed == TRUE)
   {
	   return ( FAIL );
   }
   else
   {
	   return ( PASS );
   }
}

/*
********************************************************************************
**	FUNCTION	VerifyIM_Ready
**
**	Purpose		Isolated function to request SID 1 PID 1 and check IM Ready
********************************************************************************
*/
STATUS VerifyIM_Ready (void)
{
    unsigned long EcuIndex;
    unsigned long SidIndex;

    SID_REQ SidReq;
    SID1 *pSid1;

    if (IsSid1PidSupported (-1, 1) == FALSE)
    {
        LogPrint("FAILURE: Sid 1 PID $01 not supported\n");
        return (FAIL);
    }

    SidReq.SID = 1;
    SidReq.NumIds = 1;
    SidReq.Ids[0] = 1;
    if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
    {
        /* There must be a response for ISO15765 protocol */
        if (gOBDList[gOBDListIndex].Protocol == ISO15765)
        {
            LogPrint("FAILURE: Sid1 PID request failed\n");
            return (FAIL);
        }
    }

    for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
    {
        if (gOBDResponse[EcuIndex].Sid1PidSize != 0)
        {
            break;
        }
    }

    if (EcuIndex >= gOBDNumEcus)
    {
        LogPrint("FAILURE: No Sid1 PID data\n");
        return (FAIL);
    }

    /* Verify that all SID 1 PID data is valid */
    for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
    {
        /* If PID is supported, check it */
        if (IsSid1PidSupported (EcuIndex, 1) == TRUE)
        {
            /* Check the data to see if it is valid */
            pSid1 = (SID1 *)&gOBDResponse[EcuIndex].Sid1Pid[0];
            for (SidIndex = 0; SidIndex < (gOBDResponse[EcuIndex].Sid1PidSize / sizeof(SID1)); SidIndex++)
            {
			    if ( VerifyM01P01( pSid1, SidIndex ) != PASS )
				{
				    return( FAIL );
				}
            }
        }
    }

    return (PASS);
}

/*
********************************************************************************
**	FUNCTION	RequestSID1SupportData
**
**	Purpose		Isolated function to identify support for SID 1 PID x
********************************************************************************
*/
STATUS RequestSID1SupportData (void)
{
    unsigned long EcuIndex;
    unsigned long IdIndex;
	unsigned long ulPIDSupport;	/* Evaluate $E0 PID support indication. */
    SID_REQ SidReq;

    /* Request SID 1 support data */
    for (IdIndex = 0x00; IdIndex < 0x100; IdIndex += 0x20)
    {
        SidReq.SID = 1;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = (unsigned char)IdIndex;
        if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
        {
            /* There must be a response to PID 0x00 */
            if (IdIndex == 0x00)
            {
                LogPrint("FAILURE: Sid1 support request failed\n");
                return (FAIL);
            }
        }

        /* Check if we need to request the next group */
        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            if (gOBDResponse[EcuIndex].Sid1PidSupport[IdIndex >> 5].IDBits[3] & 0x01)
            {
                break;
            }
        }
        if (EcuIndex >= gOBDNumEcus)
        {
            break;
        }
    }

	/* Enhance logic to verify support information if request is at upper limit of $E0 */
	if ( IdIndex == 0xE0 )
	{
		/* Init variable to no-support */
		ulPIDSupport = 0;

		/* For each ECU */
		for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
		{
			/* If MID is supported, keep looking */
			if ( ( gOBDResponse[EcuIndex].Sid1PidSupport[IdIndex >> 5].IDBits[0]		||
				   gOBDResponse[EcuIndex].Sid1PidSupport[IdIndex >> 5].IDBits[1]	    ||
				   gOBDResponse[EcuIndex].Sid1PidSupport[IdIndex >> 5].IDBits[2]	    ||
                (  gOBDResponse[EcuIndex].Sid1PidSupport[IdIndex >> 5].IDBits[3] & 0xFE ) ) != 0x00)
            {
				/* Flag as support indicated! */
				ulPIDSupport = 1;
            }
		}
		
		/* Flag as error if no support indicated in $E0 */
		if (ulPIDSupport == 0x00)
		{
			LogPrint("FAILURE: SID 1 PID $E0 support failure.  No PID support indicated!\n");
			return (FAIL);
		}
	}
	else
	{
		/*
		** Per J1699 rev 11.5 TC# 5.10.5 - Request request next
		** unsupported OBDMID-support OBDMID
		*/
		SidReq.SID = 1;
		SidReq.NumIds = 1;
		SidReq.Ids[0] = (unsigned char)IdIndex += 0x20;

		gIgnoreNoResponse = TRUE;

		if ( SidRequest(&SidReq, SID_REQ_NORMAL) == PASS )
		{
			/* J1850 & ISO9141 - No response preferred, but positive response
			** allowed
			*/
			if ( ( gOBDList[gOBDListIndex].Protocol == ISO15765 ) )
			{
		        gIgnoreNoResponse = FALSE;
				LogPrint("FAILURE: TC# 5.10.5 - Unexpected response from ECU!\n");
				return (FAIL);
			}
		}

		gIgnoreNoResponse = FALSE;
	}

    return (PASS);
}

//*****************************************************************************
//
//	Function:	IsSid1PidSupported
//
//	Purpose:	Determine if SID 1 PID x is supported on specific ECU.
//              Need to have called RequestSID1SupportData() previously.
//              If EcuIndex < 0 then check all ECUs.
//
//*****************************************************************************
//
//	DATE		MODIFICATION
//	02/10/05	Created common function for this logic.
//
//*****************************************************************************
unsigned int IsSid1PidSupported (unsigned int EcuIndex, unsigned int PidIndex)
{
    int index1;
    int index2;
    int mask;

    if (PidIndex == 0)
        return TRUE;            // all modules must support SID 01 PID 00

    PidIndex--;

    index1 =  PidIndex >> 5;
    index2 = (PidIndex >> 3) & 0x03;
    mask   = 0x80 >> (PidIndex & 0x07);

    if ((signed int)EcuIndex < 0)
    {
        for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
        {
            if (gOBDResponse[EcuIndex].Sid1PidSupport[index1].IDBits[index2] & mask)
                return TRUE;
        }
    }
    else
    {
        if (gOBDResponse[EcuIndex].Sid1PidSupport[index1].IDBits[index2] & mask)
            return TRUE;
    }

    return FALSE;
}

//*****************************************************************************
//
//	Function:	DetermineVariablePidSize
//
//	Purpose:	Determine number of data bytes in PIDs $06 - $09, $55 - $58
//
//*****************************************************************************
STATUS DetermineVariablePidSize (void)
{
    SID_REQ SidReq;

    SID1    *pPid1;
    unsigned char pid[OBD_MAX_ECUS];
    unsigned long EcuIdIndex[OBD_MAX_ECUS];

    unsigned long EcuIndex, numResp;

    /* only need to check once */
    if (gSid1VariablePidSize != 0)
        return PASS;

    /* -1 ==> cannot determine the PID size */
    gSid1VariablePidSize = -1;

    /* only check if needed */
    if ((IsSid1PidSupported (-1, 0x06) == FALSE) &&
        (IsSid1PidSupported (-1, 0x07) == FALSE) &&
        (IsSid1PidSupported (-1, 0x08) == FALSE) &&
        (IsSid1PidSupported (-1, 0x09) == FALSE) &&
        (IsSid1PidSupported (-1, 0x55) == FALSE) &&
        (IsSid1PidSupported (-1, 0x56) == FALSE) &&
        (IsSid1PidSupported (-1, 0x57) == FALSE) &&
        (IsSid1PidSupported (-1, 0x55) == FALSE) )
        return PASS;

    /* cannot support both PID $13 and $1D */
    if (IsSid1PidSupported (-1, 0x13) == TRUE &&
        IsSid1PidSupported (-1, 0x1D) == TRUE)
    {
        LogPrint ("FAILURE: Both PID $13 and $1D are supported\n");
        return FAIL;
    }

    /* check PID $13 first */
    if (IsSid1PidSupported (-1, 0x13) == TRUE)
    {
        SidReq.SID = 1;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = 0x13;

        if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
        {
            /* There must be a response for ISO15765 protocol */
            if (gOBDList[gOBDListIndex].Protocol == ISO15765)
            {
                LogPrint("FAILURE: Sid1 PID $13 request failed\n");
                return FAIL;
            }
        }

        numResp = 0;
        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            if (IsSid1PidSupported (EcuIndex, 0x13) == TRUE)
            {
                if (gOBDResponse[EcuIndex].Sid1PidSize > 0)
                {
                    pPid1 = (SID1 *)(gOBDResponse[EcuIndex].Sid1Pid);
                    if (pPid1->PID == 0x13)
                    {
                        EcuIdIndex[numResp] = EcuIndex;
                        pid[numResp++] = pPid1->Data[0];
                    }
                }
            }
        }

        if (numResp == 0 || numResp > 2)
        {
            LogPrint("FAILURE: Sid1 PID $13 supported by %d ECUs\n", numResp);
            return FAIL;
        }

        gSid1VariablePidSize = 1;
        return PASS;
    }

    /* check PID $1D second */
    if (IsSid1PidSupported (-1, 0x1D) == TRUE)
    {
        SidReq.SID = 1;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = 0x1D;

        if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
        {
            /* There must be a response for ISO15765 protocol */
            if (gOBDList[gOBDListIndex].Protocol == ISO15765)
            {
                LogPrint("FAILURE: Sid1 PID $1D request failed\n");
                return FAIL;
            }
        }

        numResp = 0;
        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            if (IsSid1PidSupported (EcuIndex, 0x1D) == TRUE)
            {
                if (gOBDResponse[EcuIndex].Sid1PidSize > 0)
                {
                    pPid1 = (SID1 *)(gOBDResponse[EcuIndex].Sid1Pid);
                    if (pPid1->PID == 0x1D)
                    {
                        EcuIdIndex[numResp] = EcuIndex;
                        pid[numResp++] = pPid1->Data[0];
                    }
                }
            }
        }

        if (numResp == 1)
        {
            gSid1VariablePidSize = (pid[0] == 0xff) ? 2 : 1;
            return PASS;
        }

        if (numResp == 2)
        {
            if ( (pid[0] | pid[1]) != 0xff)
            {
                gSid1VariablePidSize = 1;
                return PASS;
            }

            if ( (pid[0] == 0xf0 && pid[1] == 0x0f) ||
                 (pid[0] == 0x0f && pid[1] == 0xf0) )
            {
                gSid1VariablePidSize = 2;
                return PASS;
            }

            if ( (pid[0] == 0xcc && pid[1] == 0x33) ||
                 (pid[0] == 0x33 && pid[1] == 0xcc) )
            {
                gSid1VariablePidSize = 2;
                return PASS;
            }

            LogPrint ("FAILURE: Unable to determine number of data bytes in PIDs $06 - $09\n");
            LogPrint ("INFORMATION: ECU %X : SID $01 PID $1D -> %02X\n", GetEcuId (EcuIdIndex[0]), pid[0]);
            LogPrint ("INFORMATION: ECU %X : SID $01 PID $1D -> %02X\n", GetEcuId (EcuIdIndex[1]), pid[1]);
            return FAIL;
        }

        /* numResp == 0 or numResp > 2 */
        LogPrint("FAILURE:  Sid1 PID $1D supported by %d ECUs\n", numResp);
        return FAIL;
    }

   LogPrint ("FAILURE: Neither PID $13 nor $1D are supported\n");
   return FAIL;
}

//*****************************************************************************
//
//	Function:	GetPid4FArray
//
//	Purpose:	copy to PID $4F values into an array.
//
//*****************************************************************************
STATUS GetPid4FArray (void)
{
    SID_REQ SidReq;
    SID1 *pPid1;
    unsigned int EcuIndex;

    if (IsSid1PidSupported (-1, 0x4F) == TRUE)
    {
        SidReq.SID = 1;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = 0x4F;

        if (SidRequest(&SidReq, SID_REQ_NORMAL) != PASS)
        {
            LogPrint("FAILURE: Sid1 PID $4F request failed\n");
            return (FAIL);
        }

        for (EcuIndex = 0; EcuIndex < gOBDNumEcus; EcuIndex++)
        {
            if (IsSid1PidSupported (EcuIndex, 0x4F) == TRUE)
            {
                if (gOBDResponse[EcuIndex].Sid1PidSize > 0)
                {
                    pPid1 = (SID1 *)(gOBDResponse[EcuIndex].Sid1Pid);
                    if (pPid1->PID == 0x4F)
                    {
                        Pid4F[EcuIndex][0] = pPid1->Data[0];
                        Pid4F[EcuIndex][1] = pPid1->Data[1];
                        Pid4F[EcuIndex][2] = pPid1->Data[2];
                        Pid4F[EcuIndex][3] = pPid1->Data[3];
                    }
                }
            }
        }
    }
    return (PASS);
}