/*
********************************************************************************
** SAE J1699-3 Test Source Code
**
**  Copyright (C) 2002 Drew Technologies. http://j1699-3.sourceforge.net/
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
** This source code, when compiled and used with an SAE J2534-compatible pass
** thru device, is intended to run the tests described in the SAE J1699-3
** document in an automated manner.
**
** This computer program is based upon SAE Technical Report J1699,
** which is provided "AS IS"
**
** See j1699.c for details of how to build and run this test.
**
********************************************************************************
** DATE          MODIFICATION
** 04/26/04      Added comments: Corrections required for in-use performace
**               counter testing.
** 05/01/04      Renumber all test cases to reflect specification.  This section
**               has been indicated as section 11 in Draft 15.4.
********************************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <windows.h>
#include <conio.h>
#include "j2534.h"
#include "j1699.h"
#include "ScreenOutput.h"

enum IM_Status {Complete=0, Incomplete, NotSupported, Invalid};

enum IM_Status Test11IMStatus[OBD_MAX_ECUS][11];
enum IM_Status CurrentIMStatus[11];

SID1     Test11_Sid1Pid1[OBD_MAX_ECUS];

SID9INF8 Test10_9_Sid9Inf8[OBD_MAX_ECUS];
SID9INF8 Test11_5_Sid9Inf8[OBD_MAX_ECUS];
SID9INF8 Test11_11_Sid9Inf8[OBD_MAX_ECUS];

SID9INF8 Test11CurrentDisplayData[OBD_MAX_ECUS];

SID1     PreviousSid1Pid1[OBD_MAX_ECUS];

const char * szIM_Status[] = {"Complete", "Incomplete", "Not Supported", "Invalid"};

BOOL IsIM_ReadinessComplete (SID1 * pSid1);
BOOL EvaluateSid9Inf8 (int EcuIndex, int test_stage, BOOL bDisplayErrorMsg);
STATUS RunDynamicTest11 (void);

//-----------------------------------------------------------------------------
// Dynamic Test 11.x screen
//-----------------------------------------------------------------------------
#define COL1            1
#define COL2            25
#define COL3            40
#define COL4            60
#define COL5            65
#define COL6            70
#define COL7            75

#define COL_ECU         (COL1+15)
#define SPACE           7

#define STATUS_ROW      6
#define PRESS_ESC_ROW   21

#define STRING_WIDTH    13
#define NUMERIC_WIDTH   5
#define HEX_WIDTH       4

StaticTextElement  _string_elements11[] = 
{
   // ECU List
   {"ECU List:",   COL1, 2},
   {"ECU Status:", COL1, 3},
   {"1", COL_ECU + 0 * SPACE, 1},
   {"2", COL_ECU + 1 * SPACE, 1},
   {"3", COL_ECU + 2 * SPACE, 1},
   {"4", COL_ECU + 3 * SPACE, 1},
   {"5", COL_ECU + 4 * SPACE, 1},
   {"6", COL_ECU + 5 * SPACE, 1},
   {"7", COL_ECU + 6 * SPACE, 1},
   {"8", COL_ECU + 7 * SPACE, 1},

   // column 1
   {"I/M STATUS",                COL1, STATUS_ROW+0},
   {"Oxygen Sensor Response",    COL1, STATUS_ROW+1},
   {"Oxygen Sensor Heater",      COL1, STATUS_ROW+2},
   {"Catalyst Monitor",          COL1, STATUS_ROW+3},
   {"Catalyst Heater",           COL1, STATUS_ROW+4},
   {"A/C",                       COL1, STATUS_ROW+5},
   {"Evaportive Emissions",      COL1, STATUS_ROW+6},
   {"EGR",                       COL1, STATUS_ROW+7},
   {"AIR",                       COL1, STATUS_ROW+8},
   {"Fuel Trim",                 COL1, STATUS_ROW+9},
   {"Misfire",                   COL1, STATUS_ROW+10},
   {"Comp / Comp",               COL1, STATUS_ROW+11},

   // column 2
   {"RATE BASED COUNTER",        COL3, STATUS_ROW+0},
   {"OBD Monitoring Conditions", COL3, STATUS_ROW+1},
   {"Ignition Counter",          COL3, STATUS_ROW+2},

   {"Initial",                   COL4, STATUS_ROW+4},
   {"Current",                   COL6, STATUS_ROW+4},

   {"N",                         COL4, STATUS_ROW+5},
   {"D",                         COL5, STATUS_ROW+5},

   {"N",                         COL6, STATUS_ROW+5},
   {"D",                         COL7, STATUS_ROW+5},

   // column 2 - two values
   {"Catalyst B1",               COL3, STATUS_ROW+6},
   {"Catalyst B2",               COL3, STATUS_ROW+7},
   {"Oxygen Sensor B1",          COL3, STATUS_ROW+8},
   {"Oxygen Sensor B2",          COL3, STATUS_ROW+9},
   {"EGR",                       COL3, STATUS_ROW+10},
   {"AIR",                       COL3, STATUS_ROW+11},
   {"EVAP",                      COL3, STATUS_ROW+12},

   // misc
   {"Press ESC to exit, a number to change ECU display, or F to FAIL", COL1, PRESS_ESC_ROW},
   {"", 0, PRESS_ESC_ROW+1}
};

const int _num_string_elements11 = sizeof(_string_elements11)/sizeof(_string_elements11[0]);

DynamicValueElement  _dynamic_elements11[] = 
{
   // ECUs
   {COL_ECU + 0 * SPACE, 2, HEX_WIDTH}, // ECU 1
   {COL_ECU + 1 * SPACE, 2, HEX_WIDTH}, // ECU 2
   {COL_ECU + 2 * SPACE, 2, HEX_WIDTH}, // ECU 3
   {COL_ECU + 3 * SPACE, 2, HEX_WIDTH}, // ECU 4
   {COL_ECU + 4 * SPACE, 2, HEX_WIDTH}, // ECU 5
   {COL_ECU + 5 * SPACE, 2, HEX_WIDTH}, // ECU 6
   {COL_ECU + 6 * SPACE, 2, HEX_WIDTH}, // ECU 7
   {COL_ECU + 7 * SPACE, 2, HEX_WIDTH}, // ECU 8

   // ECU Done
   {COL_ECU + 0 * SPACE, 3, HEX_WIDTH}, // ECU Status 1
   {COL_ECU + 1 * SPACE, 3, HEX_WIDTH}, // ECU Status 2
   {COL_ECU + 2 * SPACE, 3, HEX_WIDTH}, // ECU Status 3
   {COL_ECU + 3 * SPACE, 3, HEX_WIDTH}, // ECU Status 4
   {COL_ECU + 4 * SPACE, 3, HEX_WIDTH}, // ECU Status 5
   {COL_ECU + 5 * SPACE, 3, HEX_WIDTH}, // ECU Status 6
   {COL_ECU + 6 * SPACE, 3, HEX_WIDTH}, // ECU Status 7
   {COL_ECU + 7 * SPACE, 3, HEX_WIDTH}, // ECU Status 8

   // column 1
   {COL2, STATUS_ROW+1,  STRING_WIDTH}, // Oxygen Sensor Response
   {COL2, STATUS_ROW+2,  STRING_WIDTH}, // Oxygen Sensor Heater
   {COL2, STATUS_ROW+3,  STRING_WIDTH}, // Catalyst Monitor
   {COL2, STATUS_ROW+4,  STRING_WIDTH}, // Catalyst Heater
   {COL2, STATUS_ROW+5,  STRING_WIDTH}, // A/C
   {COL2, STATUS_ROW+6,  STRING_WIDTH}, // Evaportive Emissions
   {COL2, STATUS_ROW+7,  STRING_WIDTH}, // EGR
   {COL2, STATUS_ROW+8,  STRING_WIDTH}, // AIR
   {COL2, STATUS_ROW+9,  STRING_WIDTH}, // Fuel Trim
   {COL2, STATUS_ROW+10, STRING_WIDTH}, // Misfire
   {COL2, STATUS_ROW+11, STRING_WIDTH}, // Comp / Comp

   // column 2
   {COL6, STATUS_ROW+1, NUMERIC_WIDTH}, // OBD Monitoring Conditions
   {COL6, STATUS_ROW+2, NUMERIC_WIDTH}, // Ignition Counter

   // column 2 - initial values
   {COL4, STATUS_ROW+6, NUMERIC_WIDTH}, // Catalyst B1 N
   {COL5, STATUS_ROW+6, NUMERIC_WIDTH}, // Catalyst B1 D

   {COL4, STATUS_ROW+7, NUMERIC_WIDTH}, // Catalyst B2 N
   {COL5, STATUS_ROW+7, NUMERIC_WIDTH}, // Catalyst B2 D

   {COL4, STATUS_ROW+8, NUMERIC_WIDTH}, // Oxygen Sensor B1 N
   {COL5, STATUS_ROW+8, NUMERIC_WIDTH}, // Oxygen Sensor B1 D

   {COL4, STATUS_ROW+9, NUMERIC_WIDTH}, // Oxygen Sensor B2 N
   {COL5, STATUS_ROW+9, NUMERIC_WIDTH}, // Oxygen Sensor B2 D

   {COL4, STATUS_ROW+10, NUMERIC_WIDTH},// EGR N
   {COL5, STATUS_ROW+10, NUMERIC_WIDTH},// EGR D

   {COL4, STATUS_ROW+11, NUMERIC_WIDTH},// AIR N
   {COL5, STATUS_ROW+11, NUMERIC_WIDTH},// AIR D

   {COL4, STATUS_ROW+12, NUMERIC_WIDTH},// EVAP N
   {COL5, STATUS_ROW+12, NUMERIC_WIDTH},// EVAP D

   // column 2 - current values
   {COL6, STATUS_ROW+6, NUMERIC_WIDTH}, // Catalyst B1 N
   {COL7, STATUS_ROW+6, NUMERIC_WIDTH}, // Catalyst B1 D

   {COL6, STATUS_ROW+7, NUMERIC_WIDTH}, // Catalyst B2 N
   {COL7, STATUS_ROW+7, NUMERIC_WIDTH}, // Catalyst B2 D

   {COL6, STATUS_ROW+8, NUMERIC_WIDTH}, // Oxygen Sensor B1 N
   {COL7, STATUS_ROW+8, NUMERIC_WIDTH}, // Oxygen Sensor B1 D

   {COL6, STATUS_ROW+9, NUMERIC_WIDTH}, // Oxygen Sensor B2 N
   {COL7, STATUS_ROW+9, NUMERIC_WIDTH}, // Oxygen Sensor B2 D

   {COL6, STATUS_ROW+10, NUMERIC_WIDTH},// EGR N
   {COL7, STATUS_ROW+10, NUMERIC_WIDTH},// EGR D

   {COL6, STATUS_ROW+11, NUMERIC_WIDTH},// AIR N
   {COL7, STATUS_ROW+11, NUMERIC_WIDTH},// AIR D

   {COL6, STATUS_ROW+12, NUMERIC_WIDTH},// EVAP N
   {COL7, STATUS_ROW+12, NUMERIC_WIDTH} // EVAP D
};

const int _num_dynamic_elements11 = sizeof(_dynamic_elements11)/sizeof(_dynamic_elements11[0]);

#define ECU_ID_INDEX                0
#define ECU_STATUS_INDEX            8

#define O2_SENSOR_RESP_INDEX        16
#define O2_SENSOR_HEATER_INDEX      17
#define CATALYST_MONITOR_INDEX      18
#define CATALYST_HEATER_INDEX       19
#define AIR_COND_INDEX              20
#define EVAP_EMMISION_INDEX         21
#define EGR_INDEX                   22
#define AIR_INDEX                   23
#define FUEL_TRIM_INDEX             24
#define MISFIRE_INDEX               25
#define COMP_COMP_INDEX             26

#define OBD_MONITOR_COND_INDEX      27
#define IGNITION_COUNTER_INDEX      28

#define CATALYST_B1_INIT_INDEX      29
#define CATALYST_B2_INIT_INDEX      31
#define O2_SENSOR_B1_INIT_INDEX     33
#define O2_SENSOR_B2_INIT_INDEX     35
#define EGR_INIT_INDEX              37
#define AIR_INIT_INDEX              39
#define EVAP_INIT_INDEX             41

#define CATALYST_B1_CUR_INDEX       43
#define CATALYST_B2_CUR_INDEX       45
#define O2_SENSOR_B1_CUR_INDEX      47
#define O2_SENSOR_B2_CUR_INDEX      49
#define EGR_CUR_INDEX               51
#define AIR_CUR_INDEX               53
#define EVAP_CUR_INDEX              55

#define SetFieldDec(index,val)  (update_screen_dec(_dynamic_elements11,_num_dynamic_elements11,index,val))
#define SetFieldHex(index,val)  (update_screen_hex(_dynamic_elements11,_num_dynamic_elements11,index,val))
#define SetFieldText(index,val) (update_screen_text(_dynamic_elements11,_num_dynamic_elements11,index,val))

/*
*******************************************************************************
** TestToVerifyPerformanceCounters -
** Function to run test to verify performance counters
*******************************************************************************
*/
STATUS TestToVerifyPerformanceCounters(void)
{
    unsigned int  EcuIndex;
    unsigned int  IMReadinessIndex;
    STATUS        ret_code;
    BOOL          bRunTest11_2;

    SID_REQ       SidReq;
    SID1        * pSid1;

    BOOL          bTestFailed = FALSE;
    BOOL          bSubTestFailed = FALSE;

    // initialize arrays
   	memset (Test11_Sid1Pid1, 0x00, sizeof(Test11_Sid1Pid1));
   	memset (Test11_5_Sid9Inf8, 0x00, sizeof(Test11_5_Sid9Inf8));
   	memset (Test11_11_Sid9Inf8, 0x00, sizeof(Test11_11_Sid9Inf8));
   	memset (Test11CurrentDisplayData, 0x00, sizeof(Test11CurrentDisplayData));
   	memset (PreviousSid1Pid1, 0x00, sizeof(PreviousSid1Pid1));
    for (IMReadinessIndex = 0; IMReadinessIndex < 11; IMReadinessIndex++)
    {
        for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
        {
            Test11IMStatus[EcuIndex][IMReadinessIndex] = NotSupported;
        }
        CurrentIMStatus[IMReadinessIndex] = Invalid;
    }



    LogPrint ("\n\n**** Test 11.1 (Verify performance counters) ****\n\n");

    if (gOBDEngineRunning == FALSE)
    {
        LogUserPrompt ("Turn ignition to crank position and start engine.\n\n"
                       "Press enter to continue.", ENTER_PROMPT);
        gOBDEngineRunning = TRUE;
    }

    // Test 11.1
    if (VerifyLinkActive () == FAIL)
    {
        DisconnectProtocol ();

        if (ConnectProtocol () == FAIL)
        {
            LogPrint ("FAILURE: ConnectProtocol() failed\n");
            LogPrint ("**** Test 11.1 FAILED ****\n");
            return FAIL;
        }
    }

    // Test 11.1.2 - SID 1 PID 1 request
    SidReq.SID    = 1;
    SidReq.NumIds = 1;
    SidReq.Ids[0] = 1;

    if (SidRequest (&SidReq, SID_REQ_NORMAL) != PASS)
    {
        LogPrint ("FAILURE: SID 1 PID 1 request failed\n");
        LogPrint ("**** Test 11.1 FAILED ****\n");
        return FAIL;
    }

    /* check if need to run test 11.2 */
    bRunTest11_2 = FALSE;
    for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
    {
        if (IsSid1PidSupported (EcuIndex, 1) == TRUE)
        {
            pSid1 = (SID1 *)&gOBDResponse[EcuIndex].Sid1Pid[0];

            /* save for EvaluateSid9Inf8 () */
            Test11_Sid1Pid1[EcuIndex].PID = pSid1->PID;
            Test11_Sid1Pid1[EcuIndex].Data[0] = pSid1->Data[0];
            Test11_Sid1Pid1[EcuIndex].Data[1] = pSid1->Data[1];
            Test11_Sid1Pid1[EcuIndex].Data[2] = pSid1->Data[2];
            Test11_Sid1Pid1[EcuIndex].Data[3] = pSid1->Data[3];

            if (IsIM_ReadinessComplete (pSid1) == FALSE)
                bRunTest11_2 = TRUE;
        }
    }

    if (IsSid9InfSupported (-1, 8) == TRUE)
    {
        // Test 11.1.3
        SidReq.SID    = 9;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = 8;

        if (SidRequest( &SidReq, SID_REQ_NORMAL ) != PASS)
        {
            LogPrint ("FAILURE: SID 9 INF 8 request failed\n");
            LogPrint ("**** Test 11.1 FAILED ****\n");
            return FAIL;
        }

        if (LogSid9Inf8 () != PASS)
        {
            LogPrint ("FAILURE: SID 9 INF 8 data missing\n");
            LogPrint ("**** Test 11.1 FAILED ****\n");
            return FAIL;
        }

        /* Retrieve Sid 9 Inf 8 from the frist run of test 10.9 */
        ReadSid9Inf8FromLogFile ("**** Test 10.9 ****", 
                                 "**** Test 10.9", 
                                 Test10_9_Sid9Inf8);

        if (bRunTest11_2 == FALSE)
        {
            for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
            {
                GetSid9Inf8Data (EcuIndex, &Test11_5_Sid9Inf8[EcuIndex]);

                if (EvaluateSid9Inf8 (EcuIndex, 5, FALSE) == FALSE)
                {
                    bRunTest11_2 = TRUE;
                    break;
                }
            }
        }
    }

    LogPrint ("**** Test 11.1 PASSED ****\n");

    // Test 11.2 
    LogPrint ("\n\n**** Test 11.2 ****\n");

    if (bRunTest11_2 == TRUE)
    {
        // tester-present message should already be active

        LogPrint ("INSTRUCTIONS:\n"
                  "Drive the vehicle in the manufactured-specified manner to complete all the\n"
                  "OBD monitors required to set all the supported I/M Readiness bits to a \"Ready\"\n"
                  "condition. The vehcile may have to \"soak\" with the ignition off (engine off).\n"
                  "Allow vehicle to soak according to the manufracturer-specified conditions in\n"
                  "order to run any engine-off diagnositics and/or prepare the vehicle for any \n"
                  "engine-running diagnostics on the next drive cycle that requires an engine-off\n"
                  "soak period.\n\n");

        if (LogUserPrompt ("Would you like to use the J1699 software as a monitor to view\n"
                           "the status of the I/M Readiness bits?\n"
                           "(Enter 'N' to exit the software if you are going to turn the vehicle off\n"
                           "or don't need a monitor. You can return to this point, at any time,\n"
                           "by re-starting the J1699 software and selecting 'Dynamic Tests'.)\n", YES_NO_PROMPT) != 'Y')
        {
            LogPrint ("**** Test 11.2 INCOMPLETE ****\n");
            return ABORT;
        }

        // stop tester-present message
        StopPeriodicMsg (TRUE);
		Sleep (gOBDRequestDelay);
        LogPrint ("INFORMATION: Stop periodic messages\n");

        ErrorFlags (ER_BYPASS_USER_PROMPT | ER_CONTINUE);
        ErrorCount();   /* clear error count */

        gSuspendLogOutput = TRUE;
        gSuspendScreenOutput = TRUE;
        ret_code = RunDynamicTest11 ();
        gSuspendScreenOutput = FALSE;
        gSuspendLogOutput = FALSE;
 
        ErrorFlags (0);

        // re-start tester-present message
        StartPeriodicMsg ();

        if (ret_code == ABORT)
        {
            if (ErrorCount() != 0)
            {
                LogPrint ("FAILURE: Errors detected.\n");
            }

            LogPrint ("**** Test 11.2 INCOMPLETE ****\n");
            return ret_code;
        }

        if ((ErrorCount() == 0) && (ret_code == PASS))
        {
            LogPrint ("**** Test 11.2 PASSED ****\n");
        }
        else
        {
            bTestFailed = TRUE;
            LogPrint("**** Test 11.2 FAILED ****\n");
            if (LogUserPrompt("\nErrors detected.\nDo you wish to continue?\n", YES_NO_PROMPT) == 'N')
            {
                return FAIL;
            }
        }
    }
    else
    {
        LogPrint ("**** Test 11.2 PASSED ****\n");
    }

    // Test 11.3
    LogPrint ("\n\n**** Test 11.3 ****\n");

    gOBDIMDriveCycle = TRUE;

    if (VerifyIM_Ready () != PASS)
    {
        LogPrint ("**** Test 11.3 FAILED ****\n");
        bTestFailed = TRUE;
        if (LogUserPrompt("\nErrors detected.\nDo you wish to continue?\n", YES_NO_PROMPT) == 'N')
        {
            return FAIL;
        }
    }
    else
    {
       LogPrint ("**** Test 11.3 PASSED ****\n");
    }

    // Test 11.4
    LogPrint ("\n\n**** Test 11.4 ****\n");

    if (VerifyMonitorTestSupportAndResults () != PASS)
    {
        LogPrint ("**** Test 11.4 FAILED ****\n");
        return FAIL;
    }

    LogPrint ("**** Test 11.4 PASSED ****\n");

    // Test 11.5
    LogPrint ("\n\n**** Test 11.5 ****\n");

    bSubTestFailed = FALSE;
    if (IsSid9InfSupported (-1, 8) == TRUE)
    {
        SidReq.SID    = 9;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = 8;

        if ( SidRequest( &SidReq, SID_REQ_NORMAL ) != PASS)
        {
            LogPrint ("FAILURE: SID 9 INF 8 request failed\n");
            LogPrint ("**** Test 11.5 FAILED ****\n");
            return FAIL;
        }

        if (LogSid9Inf8 () != PASS)
        {
            LogPrint ("FAILURE: SID 9 INF 8 data missing\n");
            LogPrint ("**** Test 11.5 FAILED ****\n");
            return FAIL;
        }

        for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
        {
            GetSid9Inf8Data (EcuIndex, &Test11_5_Sid9Inf8[EcuIndex]);
        }

        for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
        {
            if (EvaluateSid9Inf8 (EcuIndex, 5, TRUE) == FALSE)
            {
                 LogPrint ("**** Test 11.5 FAILED ****\n");
                 bSubTestFailed = TRUE;
                 bTestFailed = TRUE;
                 if (LogUserPrompt("\nErrors detected.\nDo you wish to continue?\n", YES_NO_PROMPT) == 'N')
                 {
                     return FAIL;
                 }
                 break;
            }
        }
    }

    LogUserPrompt ("Turn ignition off (engine off) for 60 seconds.\n\n"
                   "Press enter to continue.", ENTER_PROMPT);

    DisconnectProtocol ();

    LogUserPrompt ("Turn key on without cranking or starting engine.\n\n"
                   "Press enter to continue.", ENTER_PROMPT);

    if (bSubTestFailed == FALSE)
    {
       LogPrint ("**** Test 11.5 PASSED ****\n");
    }


    // Test 11.6
    LogPrint ("\n\n**** Test 11.6 ****\n");
    if (ConnectProtocol () == FAIL)
    {
        LogPrint ("FAILURE: ConnectProtocol() failed\n");
        LogPrint ("**** Test 11.6 FAILED ****\n");
        return FAIL;
    }
    LogPrint ("**** Test 11.6 PASSED ****\n");

    // Test 11.7
    LogPrint ("\n\n**** Test 11.7 ****\n");

    if (VerifyIM_Ready () != PASS)
    {
        LogPrint ("**** Test 11.7 FAILED ****\n");
        bTestFailed = TRUE;
        if (LogUserPrompt("\nErrors detected.\nDo you wish to continue?\n", YES_NO_PROMPT) == 'N')
        {
            return FAIL;
        }
    }
    else
    {
       LogPrint ("**** Test 11.7 PASSED ****\n");
    }

    // Test 11.8
    LogPrint ("\n\n**** Test 11.8 ****\n");

    if (VerifyDTCStoredData () != PASS)
    {
        LogPrint ("**** Test 11.8 FAILED ****\n");
        return FAIL;
    }

    LogPrint ("**** Test 11.8 PASSED ****\n");

    // Test 11.9
    LogPrint ("\n\n**** Test 11.9 ****\n");

    if (VerifyDTCPendingData () != PASS)
    {
        LogPrint ("**** Test 11.9 FAILED ****\n");
        return FAIL;
    }

    LogPrint ("**** Test 11.9 PASSED ****\n");

    // Test 11.10
    LogPrint ("\n\n**** Test 11.10 ****\n");

    if (ClearCodes () != PASS)
    {
        LogPrint ("**** Test 11.10 FAILED ****\n");
        return FAIL;
    }

    LogPrint ("**** Test 11.10 PASSED ****\n");

    // Test 11.11
    LogPrint ("\n\n**** Test 11.11 ****\n");

    if (IsSid9InfSupported (-1, 8) == TRUE)
    {
        SidReq.SID    = 9;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = 8;

        if (SidRequest( &SidReq, SID_REQ_NORMAL ) != PASS)
        {
            LogPrint ("FAILURE: SID 9 INF 8 request failed\n");
            LogPrint ("**** Test 11.11 FAILED ****\n");
            return FAIL;
        }

        if (LogSid9Inf8 () != PASS)
        {
            LogPrint ("FAILURE: SID 9 INF 8 data missing\n");
            LogPrint ("**** Test 11.11 FAILED ****\n");
            return FAIL;
        }

        for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
        {
            GetSid9Inf8Data (EcuIndex, &Test11_11_Sid9Inf8[EcuIndex]);

            if (EvaluateSid9Inf8 (EcuIndex, 11, TRUE) == FALSE)
            {
                LogPrint ("**** Test 11.11 FAILED ****\n");
                return FAIL;
            }
        }
    }
    else
    {
        LogPrint ("FAILURE: SID 9 INF 8 not supported\n");
        LogPrint ("**** Test 11.11 FAILED ****\n");
        return FAIL;
    }

    LogPrint ("**** Test 11.11 PASSED ****\n");

    return (bTestFailed != TRUE) ? PASS : FAIL;
}

/*
*******************************************************************************
** IsIM_ReadinessComplete
**
**  supported bits :: 1 -> supported,    0 -> not supported
**
**    status  bits :: 1 -> not complete, 0 -> complete (or not applicable)
**
*******************************************************************************
*/
BOOL IsIM_ReadinessComplete (SID1 * pSid1)
{
    if (pSid1->PID != 1)
        return FALSE;

    //     supported bits    status bits
    if ( ((pSid1->Data[1] & (pSid1->Data[1] >> 4)) & 0x07) != 0)
        return FALSE;

    //    supported bits   status bits
    if ( (pSid1->Data[2] & pSid1->Data[3]) != 0)
        return FALSE;

    return TRUE;
}

/*
*******************************************************************************
** EvaluateSid9Inf8
*******************************************************************************
*/
BOOL EvaluateSid9Inf8 (int EcuIndex, int test_stage, BOOL bDisplayErrorMsg)
{
    // Test 11.5
    if (test_stage == 5)
    {
        // IGNCNTR must be >= OBDCOND
        if (Test11_5_Sid9Inf8[EcuIndex].IGNCNTR < Test11_5_Sid9Inf8[EcuIndex].OBDCOND)
        {
            if (bDisplayErrorMsg == TRUE)
                LogPrint ("FAILURE: IGNCNTR less than OBDCOND\n");
            return FALSE;
        }

        // OBDCOND must be >= other monitor condition counters
        if ( (Test11_5_Sid9Inf8[EcuIndex].OBDCOND < Test11_5_Sid9Inf8[EcuIndex].CATCOND1) ||
             (Test11_5_Sid9Inf8[EcuIndex].OBDCOND < Test11_5_Sid9Inf8[EcuIndex].CATCOND2) ||
             (Test11_5_Sid9Inf8[EcuIndex].OBDCOND < Test11_5_Sid9Inf8[EcuIndex].EGRCOND)  ||
             (Test11_5_Sid9Inf8[EcuIndex].OBDCOND < Test11_5_Sid9Inf8[EcuIndex].EVAPCOND) ||
             (Test11_5_Sid9Inf8[EcuIndex].OBDCOND < Test11_5_Sid9Inf8[EcuIndex].O2COND1)  ||
             (Test11_5_Sid9Inf8[EcuIndex].OBDCOND < Test11_5_Sid9Inf8[EcuIndex].O2COND2)  ||
             (Test11_5_Sid9Inf8[EcuIndex].OBDCOND < Test11_5_Sid9Inf8[EcuIndex].AIRCOND) )
        {
            if (bDisplayErrorMsg == TRUE)
                LogPrint ("FAILURE: OBDCOND not greater than other monitor condition counters\n");
            return FALSE;
        }

        // IGNCNTR must be greater than the value in test 10.9
        if ((Test11_5_Sid9Inf8[EcuIndex].Flags != 0) &&
            (Test10_9_Sid9Inf8[EcuIndex].Flags != 0) )
        {
            int count = Test11_5_Sid9Inf8[EcuIndex].IGNCNTR - Test10_9_Sid9Inf8[EcuIndex].IGNCNTR;
            if (count < 1)
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: IGNCNTR incremented less than 1 since Test 10.9\n");
                return FALSE;
            }
        }

        // OBDCOND must have incremented by at least 1 since test 10.9
        if ((Test11_5_Sid9Inf8[EcuIndex].Flags != 0) && 
            (Test10_9_Sid9Inf8[EcuIndex].Flags != 0) )
        {
            int count = Test11_5_Sid9Inf8[EcuIndex].OBDCOND - Test10_9_Sid9Inf8[EcuIndex].OBDCOND;
            if (count < 1)
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: OBDCOND incremented less than 1 since Test 10.9\n");
                return FALSE;
            }
        }

        // if CAT monitoring not supported, must be zero
        if ((Test11_Sid1Pid1[EcuIndex].Data[2] & 0x03) == 0)
        {
            if ((Test11_5_Sid9Inf8[EcuIndex].CATCOMP1 != 0) ||
                (Test11_5_Sid9Inf8[EcuIndex].CATCOMP2 != 0) ||
                (Test11_5_Sid9Inf8[EcuIndex].CATCOND1 != 0) ||
                (Test11_5_Sid9Inf8[EcuIndex].CATCOND2 != 0) )
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: CATCOMP1, CATCOMP2, CATCOND1, CATCOND2 not supported, must be zero\n");
                return FALSE;
            }
        }

        // if EVAP monitoring not supported, must be zero
        if ((Test11_Sid1Pid1[EcuIndex].Data[2] & 0x04) == 0)
        {
            if ((Test11_5_Sid9Inf8[EcuIndex].EVAPCOMP != 0) ||
                (Test11_5_Sid9Inf8[EcuIndex].EVAPCOND != 0) )
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: EVAPCOMP and EVAPCOND not supported, must be zero\n");
                return FALSE;
            }
        }

        // if AIR monitoring not supported, must be zero
        if ((Test11_Sid1Pid1[EcuIndex].Data[2] & 0x08) == 0)
        {
            if ((Test11_5_Sid9Inf8[EcuIndex].AIRCOMP != 0) ||
                (Test11_5_Sid9Inf8[EcuIndex].AIRCOND != 0) )
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: AIRCOMP and AIRCOND not supported, must be zero\n");
                return FALSE;
            }
        }

        // if O2 monitoring not supported, must be zero
        if ((Test11_Sid1Pid1[EcuIndex].Data[2] & 0x20) == 0)
        {
            if ((Test11_5_Sid9Inf8[EcuIndex].O2COMP1 != 0) ||
                (Test11_5_Sid9Inf8[EcuIndex].O2COMP2 != 0) ||
                (Test11_5_Sid9Inf8[EcuIndex].O2COND1 != 0) ||
                (Test11_5_Sid9Inf8[EcuIndex].O2COND2 != 0) )
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: O2COMP1, O2COMP2, O2COND1, O2COND2 not supported, must be zero\n");
                return FALSE;
            }
        }

        // if EGR monitor supported, EGRCOMP, EGRCOND must have incremented since test 10.9
        if (Test11_Sid1Pid1[EcuIndex].Data[2] & 0x80)
        {
            int count = Test11_5_Sid9Inf8[EcuIndex].EGRCOMP - Test10_9_Sid9Inf8[EcuIndex].EGRCOMP;
            if (count < 1)
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: EGRCOMP incremented less than 1 since Test 10.9\n");
                return FALSE;
            }

            count = Test11_5_Sid9Inf8[EcuIndex].EGRCOND - Test10_9_Sid9Inf8[EcuIndex].EGRCOND;
            if (count < 1)
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: EGRCOND incremented less than 1 since Test 10.9\n");
                return FALSE;
            }
        }

        // if Secondary Air monitor supported, AIRCOMP, AIRCOND must have incremented since test 10.9
        if (Test11_Sid1Pid1[EcuIndex].Data[2] & 0x08)
        {
            int count = Test11_5_Sid9Inf8[EcuIndex].AIRCOMP - Test10_9_Sid9Inf8[EcuIndex].AIRCOMP;
            if (count < 1)
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: AIRCOMP incremented less than 1 since Test 10.9\n");
                return FALSE;
            }

            count = Test11_5_Sid9Inf8[EcuIndex].AIRCOND - Test10_9_Sid9Inf8[EcuIndex].AIRCOND;
            if (count < 1)
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: AIRCOND incremented less than 1 since Test 10.9\n");
                return FALSE;
            }
        }

        // if  Evap monitor supported, EVAPCOMP, EVAPCOND must have incremented since test 10.9
        if (Test11_Sid1Pid1[EcuIndex].Data[2] & 0x04)
        {
            int count = Test11_5_Sid9Inf8[EcuIndex].EVAPCOMP - Test10_9_Sid9Inf8[EcuIndex].EVAPCOMP;
            if (count < 1)
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: EVAPCOMP incremented less than 1 since Test 10.9\n");
                return FALSE;
            }

            count = Test11_5_Sid9Inf8[EcuIndex].EVAPCOND - Test10_9_Sid9Inf8[EcuIndex].EVAPCOND;
            if (count < 1)
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: EVAPCOND incremented less than 1 since Test 10.9\n");
                return FALSE;
            }
        }

        // if CAT monitoring is supported, at least one bank must have increased
        if ((Test11_Sid1Pid1[EcuIndex].Data[2] & 0x03) != 0)
        {
            int count1 = Test11_5_Sid9Inf8[EcuIndex].CATCOMP1 - Test10_9_Sid9Inf8[EcuIndex].CATCOMP1;
            int count2 = Test11_5_Sid9Inf8[EcuIndex].CATCOND1 - Test10_9_Sid9Inf8[EcuIndex].CATCOND1;

            int count3 = Test11_5_Sid9Inf8[EcuIndex].CATCOMP2 - Test10_9_Sid9Inf8[EcuIndex].CATCOMP2;
            int count4 = Test11_5_Sid9Inf8[EcuIndex].CATCOND2 - Test10_9_Sid9Inf8[EcuIndex].CATCOND2;

            if ( (count1 < 1 || count2 < 1) && (count3 < 1 || count4 < 1) )
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: Neither bank of the CAT monitor incremented\n");
                return FALSE;
            }
        }

        // if O2 monitoring is supported, at least one bank must have increased
        if ((Test11_Sid1Pid1[EcuIndex].Data[2] & 0x20) != 0)
        {
            int count1 = Test11_5_Sid9Inf8[EcuIndex].O2COMP1 - Test10_9_Sid9Inf8[EcuIndex].O2COMP1;
            int count2 = Test11_5_Sid9Inf8[EcuIndex].O2COND1 - Test10_9_Sid9Inf8[EcuIndex].O2COND1;

            int count3 = Test11_5_Sid9Inf8[EcuIndex].O2COMP2 - Test10_9_Sid9Inf8[EcuIndex].O2COMP2;
            int count4 = Test11_5_Sid9Inf8[EcuIndex].O2COND2 - Test10_9_Sid9Inf8[EcuIndex].O2COND2;

            if ( (count1 < 1 || count2 < 1) && (count3 < 1 || count4 < 1) )
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: Neither bank of the O2 monitor incremented\n");
                return FALSE;
            }
        }
    }

    // Test 11.11
    if (test_stage == 11)
    {
        // if not supported, then cannot do comparsions
        if (IsSid9InfSupported (EcuIndex, 8) == TRUE)
        {
            // IGNCNTR must be >= OBDCOND
            if (Test11_11_Sid9Inf8[EcuIndex].IGNCNTR < Test11_11_Sid9Inf8[EcuIndex].OBDCOND)
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: IGNCNTR less than OBDCOND\n");
                return FALSE;
            }

            // OBDCONF must be >= other monitor condition counters
            if ( (Test11_11_Sid9Inf8[EcuIndex].OBDCOND < Test11_11_Sid9Inf8[EcuIndex].CATCOND1) ||
                 (Test11_11_Sid9Inf8[EcuIndex].OBDCOND < Test11_11_Sid9Inf8[EcuIndex].CATCOND2) ||
                 (Test11_11_Sid9Inf8[EcuIndex].OBDCOND < Test11_11_Sid9Inf8[EcuIndex].EGRCOND)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].OBDCOND < Test11_11_Sid9Inf8[EcuIndex].EVAPCOND) ||
                 (Test11_11_Sid9Inf8[EcuIndex].OBDCOND < Test11_11_Sid9Inf8[EcuIndex].O2COND1)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].OBDCOND < Test11_11_Sid9Inf8[EcuIndex].O2COND2) )
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: OBDCOND not greater than other monitor condition counters\n");
                return FALSE;
            }

            // OBD condition counters must be the same as in test 11.5
            if ( (Test11_11_Sid9Inf8[EcuIndex].OBDCOND  != Test11_5_Sid9Inf8[EcuIndex].OBDCOND)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].IGNCNTR  != Test11_5_Sid9Inf8[EcuIndex].IGNCNTR)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].CATCOMP1 != Test11_5_Sid9Inf8[EcuIndex].CATCOMP1) ||
                 (Test11_11_Sid9Inf8[EcuIndex].CATCOND1 != Test11_5_Sid9Inf8[EcuIndex].CATCOND1) ||
                 (Test11_11_Sid9Inf8[EcuIndex].CATCOMP2 != Test11_5_Sid9Inf8[EcuIndex].CATCOMP2) ||
                 (Test11_11_Sid9Inf8[EcuIndex].CATCOND2 != Test11_5_Sid9Inf8[EcuIndex].CATCOND2) ||
                 (Test11_11_Sid9Inf8[EcuIndex].O2COMP1  != Test11_5_Sid9Inf8[EcuIndex].O2COMP1)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].O2COMP2  != Test11_5_Sid9Inf8[EcuIndex].O2COMP2)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].O2COND1  != Test11_5_Sid9Inf8[EcuIndex].O2COND1)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].O2COND2  != Test11_5_Sid9Inf8[EcuIndex].O2COND2)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].EGRCOMP  != Test11_5_Sid9Inf8[EcuIndex].EGRCOMP)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].EGRCOND  != Test11_5_Sid9Inf8[EcuIndex].EGRCOND)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].AIRCOMP  != Test11_5_Sid9Inf8[EcuIndex].AIRCOMP)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].AIRCOND  != Test11_5_Sid9Inf8[EcuIndex].AIRCOND)  ||
                 (Test11_11_Sid9Inf8[EcuIndex].EVAPCOMP != Test11_5_Sid9Inf8[EcuIndex].EVAPCOMP) ||
                 (Test11_11_Sid9Inf8[EcuIndex].EVAPCOND != Test11_5_Sid9Inf8[EcuIndex].EVAPCOND) )
            {
                if (bDisplayErrorMsg == TRUE)
                    LogPrint ("FAILURE: OBD counters different from values in test 11.5\n");
                return FALSE;
            }
        }
    }

    return TRUE;
}

/*
*******************************************************************************
** SelectECU
*******************************************************************************
*/
void SelectECU (int new_index, int old_index)
{
   int index;

   // un-highlite previous selection
   setrgb (-1);
   index = ECU_ID_INDEX + old_index;
   if ((ECU_ID_INDEX <= index) && (index <= ECU_ID_INDEX+OBD_MAX_ECUS))
      SetFieldHex (index, GetEcuId(old_index));

   // highlite new selection
   setrgb (6);
   index = ECU_ID_INDEX + new_index;
   if ((ECU_ID_INDEX <= index) && (index <= ECU_ID_INDEX+OBD_MAX_ECUS))
      SetFieldHex (index, GetEcuId(new_index));

   // restore screen attributes
   setrgb (-1);
}

/*
*******************************************************************************
** DisplayEcuData
*******************************************************************************
*/
void DisplayEcuData (int EcuIndex)
{
    int index;

    // IM Status
    SetFieldText (O2_SENSOR_RESP_INDEX,   szIM_Status[Test11IMStatus[EcuIndex][0]]);
    SetFieldText (O2_SENSOR_HEATER_INDEX, szIM_Status[Test11IMStatus[EcuIndex][1]]);
    SetFieldText (CATALYST_MONITOR_INDEX, szIM_Status[Test11IMStatus[EcuIndex][2]]);
    SetFieldText (CATALYST_HEATER_INDEX,  szIM_Status[Test11IMStatus[EcuIndex][3]]);
    SetFieldText (AIR_COND_INDEX,         szIM_Status[Test11IMStatus[EcuIndex][4]]);
    SetFieldText (EVAP_EMMISION_INDEX,    szIM_Status[Test11IMStatus[EcuIndex][5]]);
    SetFieldText (EGR_INDEX,              szIM_Status[Test11IMStatus[EcuIndex][6]]);
    SetFieldText (AIR_INDEX,              szIM_Status[Test11IMStatus[EcuIndex][7]]);
    SetFieldText (FUEL_TRIM_INDEX,        szIM_Status[Test11IMStatus[EcuIndex][8]]);
    SetFieldText (MISFIRE_INDEX,          szIM_Status[Test11IMStatus[EcuIndex][9]]);
    SetFieldText (COMP_COMP_INDEX,        szIM_Status[Test11IMStatus[EcuIndex][10]]);

    // copy for active ECU
    for (index=0; index<11; index++)
        CurrentIMStatus[index] = Test11IMStatus[EcuIndex][index];

    // Rate based counter
    SetFieldDec (OBD_MONITOR_COND_INDEX, Test11CurrentDisplayData[EcuIndex].OBDCOND);
    SetFieldDec (IGNITION_COUNTER_INDEX, Test11CurrentDisplayData[EcuIndex].IGNCNTR);

    // Rate based counter - initial values
    SetFieldDec (CATALYST_B1_INIT_INDEX,   Test10_9_Sid9Inf8[EcuIndex].CATCOMP1);
    SetFieldDec (CATALYST_B1_INIT_INDEX+1, Test10_9_Sid9Inf8[EcuIndex].CATCOND1);

    SetFieldDec (CATALYST_B2_INIT_INDEX,   Test10_9_Sid9Inf8[EcuIndex].CATCOMP2);
    SetFieldDec (CATALYST_B2_INIT_INDEX+1, Test10_9_Sid9Inf8[EcuIndex].CATCOND2);

    SetFieldDec (O2_SENSOR_B1_INIT_INDEX,   Test10_9_Sid9Inf8[EcuIndex].O2COMP1);
    SetFieldDec (O2_SENSOR_B1_INIT_INDEX+1, Test10_9_Sid9Inf8[EcuIndex].O2COND1);

    SetFieldDec (O2_SENSOR_B2_INIT_INDEX,   Test10_9_Sid9Inf8[EcuIndex].O2COMP2);
    SetFieldDec (O2_SENSOR_B2_INIT_INDEX+1, Test10_9_Sid9Inf8[EcuIndex].O2COND2);

    SetFieldDec (EGR_INIT_INDEX,   Test10_9_Sid9Inf8[EcuIndex].EGRCOMP);
    SetFieldDec (EGR_INIT_INDEX+1, Test10_9_Sid9Inf8[EcuIndex].EGRCOND);

    SetFieldDec (AIR_INIT_INDEX,   Test10_9_Sid9Inf8[EcuIndex].AIRCOMP);
    SetFieldDec (AIR_INIT_INDEX+1, Test10_9_Sid9Inf8[EcuIndex].AIRCOND);

    SetFieldDec (EVAP_INIT_INDEX,   Test10_9_Sid9Inf8[EcuIndex].EVAPCOMP);
    SetFieldDec (EVAP_INIT_INDEX+1, Test10_9_Sid9Inf8[EcuIndex].EVAPCOND);

    // Rate based counter - current values
    SetFieldDec (CATALYST_B1_CUR_INDEX,   Test11CurrentDisplayData[EcuIndex].CATCOMP1);
    SetFieldDec (CATALYST_B1_CUR_INDEX+1, Test11CurrentDisplayData[EcuIndex].CATCOND1);

    SetFieldDec (CATALYST_B2_CUR_INDEX,   Test11CurrentDisplayData[EcuIndex].CATCOMP2);
    SetFieldDec (CATALYST_B2_CUR_INDEX+1, Test11CurrentDisplayData[EcuIndex].CATCOND2);

    SetFieldDec (O2_SENSOR_B1_CUR_INDEX,   Test11CurrentDisplayData[EcuIndex].O2COMP1);
    SetFieldDec (O2_SENSOR_B1_CUR_INDEX+1, Test11CurrentDisplayData[EcuIndex].O2COND1);

    SetFieldDec (O2_SENSOR_B2_CUR_INDEX,   Test11CurrentDisplayData[EcuIndex].O2COMP2);
    SetFieldDec (O2_SENSOR_B2_CUR_INDEX+1, Test11CurrentDisplayData[EcuIndex].O2COND2);

    SetFieldDec (EGR_CUR_INDEX,   Test11CurrentDisplayData[EcuIndex].EGRCOMP);
    SetFieldDec (EGR_CUR_INDEX+1, Test11CurrentDisplayData[EcuIndex].EGRCOND);

    SetFieldDec (AIR_CUR_INDEX,   Test11CurrentDisplayData[EcuIndex].AIRCOMP);
    SetFieldDec (AIR_CUR_INDEX+1, Test11CurrentDisplayData[EcuIndex].AIRCOND);

    SetFieldDec (EVAP_CUR_INDEX,   Test11CurrentDisplayData[EcuIndex].EVAPCOMP);
    SetFieldDec (EVAP_CUR_INDEX+1, Test11CurrentDisplayData[EcuIndex].EVAPCOND);
}

/*
*******************************************************************************
** SaveSid9Inf8Data
*******************************************************************************
*/
BOOL SaveSid9Inf8Data (unsigned int EcuIndex, SID9INF8 * pSid9Inf8)
{
    BOOL rc = FALSE;

    // note any differences
    if ( (Test11CurrentDisplayData[EcuIndex].OBDCOND  != pSid9Inf8->OBDCOND)  ||
         (Test11CurrentDisplayData[EcuIndex].IGNCNTR  != pSid9Inf8->IGNCNTR)  ||
         (Test11CurrentDisplayData[EcuIndex].CATCOMP1 != pSid9Inf8->CATCOMP1) ||
         (Test11CurrentDisplayData[EcuIndex].CATCOND1 != pSid9Inf8->CATCOND1) ||
         (Test11CurrentDisplayData[EcuIndex].CATCOMP2 != pSid9Inf8->CATCOMP2) ||
         (Test11CurrentDisplayData[EcuIndex].CATCOND2 != pSid9Inf8->CATCOND2) ||
         (Test11CurrentDisplayData[EcuIndex].O2COMP1  != pSid9Inf8->O2COMP1)  ||
         (Test11CurrentDisplayData[EcuIndex].O2COND1  != pSid9Inf8->O2COND1)  ||
         (Test11CurrentDisplayData[EcuIndex].O2COMP2  != pSid9Inf8->O2COMP2)  ||
         (Test11CurrentDisplayData[EcuIndex].O2COND2  != pSid9Inf8->O2COND2)  ||
         (Test11CurrentDisplayData[EcuIndex].EGRCOMP  != pSid9Inf8->EGRCOMP)  ||
         (Test11CurrentDisplayData[EcuIndex].EGRCOND  != pSid9Inf8->EGRCOND)  ||
         (Test11CurrentDisplayData[EcuIndex].AIRCOMP  != pSid9Inf8->AIRCOMP)  ||
         (Test11CurrentDisplayData[EcuIndex].AIRCOND  != pSid9Inf8->AIRCOND)  ||
         (Test11CurrentDisplayData[EcuIndex].EVAPCOMP != pSid9Inf8->EVAPCOMP) ||
         (Test11CurrentDisplayData[EcuIndex].EVAPCOND != pSid9Inf8->EVAPCOND) )
    {
        rc = TRUE;
    }

    // Rate based counter
    Test11CurrentDisplayData[EcuIndex].OBDCOND = pSid9Inf8->OBDCOND;
    Test11CurrentDisplayData[EcuIndex].IGNCNTR = pSid9Inf8->IGNCNTR;


    // Rate based counter - current values
    Test11CurrentDisplayData[EcuIndex].CATCOMP1 = pSid9Inf8->CATCOMP1;
    Test11CurrentDisplayData[EcuIndex].CATCOND1 = pSid9Inf8->CATCOND1;
    Test11CurrentDisplayData[EcuIndex].CATCOMP2 = pSid9Inf8->CATCOMP2;
    Test11CurrentDisplayData[EcuIndex].CATCOND2 = pSid9Inf8->CATCOND2;
    Test11CurrentDisplayData[EcuIndex].O2COMP1  = pSid9Inf8->O2COMP1;
    Test11CurrentDisplayData[EcuIndex].O2COND1  = pSid9Inf8->O2COND1;
    Test11CurrentDisplayData[EcuIndex].O2COMP2  = pSid9Inf8->O2COMP2;
    Test11CurrentDisplayData[EcuIndex].O2COND2  = pSid9Inf8->O2COND2;
    Test11CurrentDisplayData[EcuIndex].EGRCOMP  = pSid9Inf8->EGRCOMP;
    Test11CurrentDisplayData[EcuIndex].EGRCOND  = pSid9Inf8->EGRCOND;
    Test11CurrentDisplayData[EcuIndex].AIRCOMP  = pSid9Inf8->AIRCOMP;
    Test11CurrentDisplayData[EcuIndex].AIRCOND  = pSid9Inf8->AIRCOND;
    Test11CurrentDisplayData[EcuIndex].EVAPCOMP = pSid9Inf8->EVAPCOMP;
    Test11CurrentDisplayData[EcuIndex].EVAPCOND = pSid9Inf8->EVAPCOND;

    return rc;
}

/*
*******************************************************************************
** UpdateSid9Inf8Display
*******************************************************************************
*/
void UpdateSid9Inf8Display (unsigned int EcuIndex, SID9INF8 * pSid9Inf8)
{
    // Rate based counter
    if (pSid9Inf8->OBDCOND != Test11CurrentDisplayData[EcuIndex].OBDCOND)
    {
        SetFieldDec (OBD_MONITOR_COND_INDEX, pSid9Inf8->OBDCOND);
    }

    if (pSid9Inf8->IGNCNTR != Test11CurrentDisplayData[EcuIndex].IGNCNTR)
    {
        SetFieldDec (IGNITION_COUNTER_INDEX, pSid9Inf8->IGNCNTR);
    }

    // Rate based counter - current values
    if (pSid9Inf8->CATCOMP1 != Test11CurrentDisplayData[EcuIndex].CATCOMP1)
    {
        SetFieldDec (CATALYST_B1_CUR_INDEX, pSid9Inf8->CATCOMP1);
    }

    if (pSid9Inf8->CATCOND1 != Test11CurrentDisplayData[EcuIndex].CATCOND1)
    {
        SetFieldDec (CATALYST_B1_CUR_INDEX+1, pSid9Inf8->CATCOND1);
    }

    if (pSid9Inf8->CATCOMP2 != Test11CurrentDisplayData[EcuIndex].CATCOMP2)
    {
        SetFieldDec (CATALYST_B2_CUR_INDEX, pSid9Inf8->CATCOMP2);
    }

    if (pSid9Inf8->CATCOND2 != Test11CurrentDisplayData[EcuIndex].CATCOND2)
    {
        SetFieldDec (CATALYST_B2_CUR_INDEX+1, pSid9Inf8->CATCOND2);
    }

    if (pSid9Inf8->O2COMP1 != Test11CurrentDisplayData[EcuIndex].O2COMP1)
    {
        SetFieldDec (O2_SENSOR_B1_CUR_INDEX, pSid9Inf8->O2COMP1);
    }

    if (pSid9Inf8->O2COND1 != Test11CurrentDisplayData[EcuIndex].O2COND1)
    {
        SetFieldDec (O2_SENSOR_B1_CUR_INDEX+1, pSid9Inf8->O2COND1);
    }

    if (pSid9Inf8->O2COMP2 != Test11CurrentDisplayData[EcuIndex].O2COMP2)
    {
        SetFieldDec (O2_SENSOR_B2_CUR_INDEX, pSid9Inf8->O2COMP2);
    }

    if (pSid9Inf8->O2COND2 != Test11CurrentDisplayData[EcuIndex].O2COND2)
    {
        SetFieldDec (O2_SENSOR_B2_CUR_INDEX+1, pSid9Inf8->O2COND2);
    }

    if (pSid9Inf8->EGRCOMP != Test11CurrentDisplayData[EcuIndex].EGRCOMP)
    {
        SetFieldDec (EGR_CUR_INDEX, pSid9Inf8->EGRCOMP);
    }

    if (pSid9Inf8->EGRCOND != Test11CurrentDisplayData[EcuIndex].EGRCOND)
    {
        SetFieldDec (EGR_CUR_INDEX+1, pSid9Inf8->EGRCOND);
    }

    if (pSid9Inf8->AIRCOMP != Test11CurrentDisplayData[EcuIndex].AIRCOMP)
    {
        SetFieldDec (AIR_CUR_INDEX, pSid9Inf8->AIRCOMP);
    }

    if (pSid9Inf8->AIRCOND != Test11CurrentDisplayData[EcuIndex].AIRCOND)
    {
        SetFieldDec (AIR_CUR_INDEX+1, pSid9Inf8->AIRCOND);
    }

    if (pSid9Inf8->EVAPCOMP != Test11CurrentDisplayData[EcuIndex].EVAPCOMP)
    {
        SetFieldDec (EVAP_CUR_INDEX, pSid9Inf8->EVAPCOMP);
    }

    if (pSid9Inf8->EVAPCOND != Test11CurrentDisplayData[EcuIndex].EVAPCOND)
    {
        SetFieldDec (EVAP_CUR_INDEX+1, pSid9Inf8->EVAPCOND);
    }
}

/*
*******************************************************************************
** SaveSid1Pid1Data
*******************************************************************************
*/
BOOL SaveSid1Pid1Data (unsigned int EcuIndex)
{
    SID1 * pSid1;
    BOOL   rc = FALSE;

    pSid1 = (SID1 *)&gOBDResponse[EcuIndex].Sid1Pid[0];

    if (pSid1->PID == 1)
    {
        // note any differences
        if ((PreviousSid1Pid1[EcuIndex].Data[0] != pSid1->Data[0]) ||
            (PreviousSid1Pid1[EcuIndex].Data[1] != pSid1->Data[1]) ||
            (PreviousSid1Pid1[EcuIndex].Data[2] != pSid1->Data[2]) ||
            (PreviousSid1Pid1[EcuIndex].Data[3] != pSid1->Data[3]) )
        {
            PreviousSid1Pid1[EcuIndex].Data[0] = pSid1->Data[0];
            PreviousSid1Pid1[EcuIndex].Data[1] = pSid1->Data[1];
            PreviousSid1Pid1[EcuIndex].Data[2] = pSid1->Data[2];
            PreviousSid1Pid1[EcuIndex].Data[3] = pSid1->Data[3];
            rc = TRUE;
        }

        // Oxygen Sensor Response
        if (pSid1->Data[2] & 1<<5)
            Test11IMStatus[EcuIndex][0] = (pSid1->Data[3] & 1<<5) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][0] = NotSupported;

        // Oxygen Sensor Heater
        if (pSid1->Data[2] & 1<<6)
            Test11IMStatus[EcuIndex][1] = (pSid1->Data[3] & 1<<6) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][1] = NotSupported;

        // Catalyst Monitor
        if (pSid1->Data[2] & 1<<0)
            Test11IMStatus[EcuIndex][2] = (pSid1->Data[3] & 1<<0) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][2] = NotSupported;

        // Catalyst Heater
        if (pSid1->Data[2] & 1<<1)
            Test11IMStatus[EcuIndex][3] = (pSid1->Data[3] & 1<<1) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][3] = NotSupported;

        // A/C
        if (pSid1->Data[2] & 1<<4)
            Test11IMStatus[EcuIndex][4] = (pSid1->Data[3] & 1<<4) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][4] = NotSupported;

        // Evaportive Emissions
        if (pSid1->Data[2] & 1<<2)
            Test11IMStatus[EcuIndex][5] = (pSid1->Data[3] & 1<<2) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][5] = NotSupported;

        // EGR
        if (pSid1->Data[2] & 1<<7)
            Test11IMStatus[EcuIndex][6] = (pSid1->Data[3] & 1<<7) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][6] = NotSupported;

        // AIR
        if (pSid1->Data[2] & 1<<3)
            Test11IMStatus[EcuIndex][7] = (pSid1->Data[3] & 1<<3) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][7] = NotSupported;

        // Fuel Trim
        if (pSid1->Data[1] & 1<<1)
            Test11IMStatus[EcuIndex][8] = (pSid1->Data[1] & 1<<5) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][8] = NotSupported;

        // Misfire
        if (pSid1->Data[1] & 1<<0)
            Test11IMStatus[EcuIndex][9] = (pSid1->Data[1] & 1<<4) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][9] = NotSupported;

        // Comp / Comp
        if (pSid1->Data[1] & 1<<2)
            Test11IMStatus[EcuIndex][10] = (pSid1->Data[1] & 1<<6) ? Incomplete : Complete;
        else
            Test11IMStatus[EcuIndex][10] = NotSupported;
    }

    return rc;
}

/*
*******************************************************************************
** UpdateSid1Pid1Display
*******************************************************************************
*/
void UpdateSid1Pid1Display (unsigned int EcuIndex)
{
    int index;

    for (index=0; index<11; index++)
    {
        if (CurrentIMStatus[index] != Test11IMStatus[EcuIndex][index])
        {
            CurrentIMStatus[index] = Test11IMStatus[EcuIndex][index];
            SetFieldText (O2_SENSOR_RESP_INDEX+index, szIM_Status[Test11IMStatus[EcuIndex][index]]);
        }
    }
}

/*
*******************************************************************************
** RunDynamicTest11
*******************************************************************************
*/
STATUS RunDynamicTest11 (void)
{
    unsigned int  TestState, bSid9Inf8Support, bLogMessage;
    unsigned long t1SecTimer, tDelayTimeStamp;
    unsigned int  EcuIndex, EcuMask, CurrentEcuIndex;
    unsigned int  IMReadyDoneFlags, Sid9Inf8DoneFlags, EcuDone;

    const unsigned int EcuDoneMask = (1 << gUserNumEcus) - 1;

    SID_REQ       SidReq;
    SID1 *        pSid1;

    EcuDone           = 0;
    IMReadyDoneFlags  = 0;
    Sid9Inf8DoneFlags = 0;

    bSid9Inf8Support  = IsSid9InfSupported (-1, 0x08);

    // initialize static text elements
    init_screen (_string_elements11, _num_string_elements11);

    for (EcuIndex=0, EcuMask=1; EcuIndex<gUserNumEcus; EcuIndex++, EcuMask<<=1)
    {
        SetFieldHex (ECU_ID_INDEX+EcuIndex, GetEcuId(EcuIndex));

        if (IsSid1PidSupported (EcuIndex, 1) == FALSE)
        {
            IMReadyDoneFlags |= EcuMask;
        }

        if (IsSid9InfSupported (EcuIndex, 8) == FALSE)
        {
            Sid9Inf8DoneFlags |= EcuMask;
        }

        if ( (IMReadyDoneFlags & Sid9Inf8DoneFlags & EcuMask) != 0)
        {
            EcuDone |= EcuMask;
            SetFieldText (ECU_STATUS_INDEX+EcuIndex, "N/A");
        }
    }

    CurrentEcuIndex = 0;

    SelectECU (CurrentEcuIndex, -1);
    DisplayEcuData (CurrentEcuIndex);

    // flush the STDIN stream of any user input before loop
    clear_keyboard_buffer ();

    tDelayTimeStamp = t1SecTimer = GetTickCount ();
    TestState = 0;

    //-------------------------------------------
    // loop until test completes
    //-------------------------------------------
    for (;;)
    {
        //-------------------------------------------
        // request SID 1 PID 1
        //-------------------------------------------
        SidReq.SID    = 1;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = 1;

        if (SidRequest (&SidReq, SID_REQ_NO_PERIODIC_DISABLE) != PASS)
        {
            LogMsgCopy ();
            LogPrint ("FAILURE: Sid 1 Pid 1 request failed\n");
            return FAIL;
        }

        bLogMessage = FALSE;
        for (EcuIndex = 0, EcuMask=1; EcuIndex < gUserNumEcus; EcuIndex++, EcuMask<<=1)
        {
            if (gOBDResponse[EcuIndex].Sid1PidSize > 0)
            {
                if (SaveSid1Pid1Data (EcuIndex) == TRUE)
                {
                    bLogMessage = TRUE;

                    if ( (IMReadyDoneFlags & EcuMask) == 0)
                    {
                        pSid1 = (SID1 *)&gOBDResponse[EcuIndex].Sid1Pid[0];
                        if (IsIM_ReadinessComplete (pSid1) == TRUE)
                            IMReadyDoneFlags |= EcuMask;
                    }
                }

                if (EcuIndex == CurrentEcuIndex)
                    UpdateSid1Pid1Display (EcuIndex);
            }
        }

        if (bLogMessage == TRUE)
            LogMsgCopy ();

        //-------------------------------------------
        // Get SID 9 INF 8
        //-------------------------------------------
        if (bSid9Inf8Support)
        {
            SidReq.SID    = 9;
            SidReq.NumIds = 1;
            SidReq.Ids[0] = 8;

            if (SidRequest (&SidReq, SID_REQ_NO_PERIODIC_DISABLE) != PASS)
            {
                LogMsgCopy ();
                LogPrint ("FAILURE: Sid 9 Inf 8 request failed\n");
                return FAIL;
            }

            bLogMessage = FALSE;
            for (EcuIndex = 0, EcuMask=1; EcuIndex < gUserNumEcus; EcuIndex++, EcuMask<<=1)
            {
                if (GetSid9Inf8Data (EcuIndex, &Test11_5_Sid9Inf8[EcuIndex]) == PASS)
                {
                    if (EcuIndex == CurrentEcuIndex)
                        UpdateSid9Inf8Display (EcuIndex, &Test11_5_Sid9Inf8[EcuIndex]);

                    if (SaveSid9Inf8Data (EcuIndex, &Test11_5_Sid9Inf8[EcuIndex]) == TRUE)
                    {
                        bLogMessage = TRUE;

                        if ( (Sid9Inf8DoneFlags & EcuMask) == 0)
                        {
                            if (EvaluateSid9Inf8 (EcuIndex, 5, FALSE) == TRUE)
                                Sid9Inf8DoneFlags |= EcuMask;
                        }
                    }
                }
            }

            if (bLogMessage == TRUE)
                LogMsgCopy ();
        }

        //-------------------------------------------
        // Check if test is complete
        //-------------------------------------------
        for (EcuIndex=0, EcuMask=1; EcuIndex < gUserNumEcus; EcuIndex++, EcuMask<<=1)
        {
            if ( (EcuDone & EcuMask) == 0)
            {
                if ( (IMReadyDoneFlags & Sid9Inf8DoneFlags & EcuMask) != 0)
                {
                    EcuDone |= EcuMask;
                    SetFieldText (ECU_STATUS_INDEX+EcuIndex, "Done");
                }
            }
        }

        if (EcuDone == EcuDoneMask)
            return PASS;

        //-------------------------------------------
        // Check for num or ESC key, delay 1 second
        //-------------------------------------------
        do
        {
            if (_kbhit () != 0)
            {
                char c = _getch ();
                if (c == 27)                    // ESC key
                {
                    LogPrint ("INFORMATION: Test 11 aborted by user\n\n");
                    return ABORT;
                }

                if ((c == 'F') || (c == 'f'))   // "FAIL" key
                {
                    LogPrint ("INFORMATION: Test 11 failed by user\n\n");
                    return FAIL;
                }

                if (('1' <= c) && (c <= '8'))   // new ECU index
                {
                    EcuIndex = c - '1';         // zero-based index
                    if (EcuIndex < gUserNumEcus && EcuIndex != CurrentEcuIndex)
                    {
                        SelectECU (EcuIndex, CurrentEcuIndex);
                        DisplayEcuData (EcuIndex);
                        CurrentEcuIndex = EcuIndex;
                    }
                }
            }

            tDelayTimeStamp = GetTickCount ();

            Sleep ( min (1000 - (tDelayTimeStamp - t1SecTimer), 50) );

        } while (tDelayTimeStamp - t1SecTimer < 1000);

        t1SecTimer = tDelayTimeStamp;
    }

    LogPrint ("FAILURE: Error in RunDynamicTest11()\n\n");
    return FAIL;
}
