/*
********************************************************************************
** SAE J1699-3 Test Source Code
**
**  Copyright (C) 2002 Drew Technologies. http://j1699-3.sourceforge.net/
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
** This source code, when compiled and used with an SAE J2534-compatible pass
** thru device, is intended to run the tests described in the SAE J1699-3
** document in an automated manner.
**
** This computer program is based upon SAE Technical Report J1699,
** which is provided "AS IS"
**
** See j1699.c for details of how to build and run this test.
**
********************************************************************************
* DATE          MODIFICATION
* 05/01/04      Renumber all test cases to reflect specification.  This section
*               has been indicated as section 10 in Draft 15.4.
********************************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <time.h>
#include <windows.h>
#include "j2534.h"
#include "j1699.h"
#include "ScreenOutput.h"

int StartLogFile (void);
STATUS RunDynamicTest10 (unsigned long tEngineStartTimeStamp);

/* Initial data for Test 11.x. Declared in TestToVerifyPerformanceCounters.c */
extern SID9INF8 Test10_9_Sid9Inf8[OBD_MAX_ECUS];

/*
*******************************************************************************
** Test 10.x dynamic screen mapping
*******************************************************************************
*/

StaticTextElement  _string_elements10[] = 
{
   {"Drive the vehicle in the following maner, at an altitude < 8000 ft", 1, 0},
   {"(BARO < 22 in Hg) and ambient temperature > or = 20 deg F,", 1, 1},
   {"so that the OBD Condition Counter will increment:", 1, 2},

   {"- Continuous time > or = 30 seconds with vehicle speed < or = 1 MPH", 1, 4},
   {"  and accelerator pedal released.", 2, 5},
   {"- Cumulative time > or = 300 seconds with vehicle speed > or = 25 MPH.", 1, 6},
   {"- Cumulative time since engine start > or = 600 seconds.", 1, 7},

   {"OBD Drive Cycle Status", 1, 9},

   {" 30 Seconds Idle Timer:", 1, 11},
   {"300 Seconds at speeds greater then 25 MPH Timer:", 1, 12},
   {"600 Seconds Total Drive Timer:", 1, 13},

   {"ECU ID:", 1, 15},

   {"Initial OBDCOND:", 1, 17},
   {"Current OBDCOND:", 1, 18},

   {"Initial IGNCTR:", 1, 20},
   {"Current IGNCTR:", 1, 21},

   {"Speed:", 60, 12},

   {"Press ESC to abort", 1, 23},
   {"", 0, 24},
};

const int _num_string_elements10 = sizeof(_string_elements10)/sizeof(_string_elements10[0]);

#define SPACE   7
DynamicValueElement  _dynamic_elements10[] = 
{
   {26, 11, 5},             // 30 seconds idle drive cycle timer
   {51, 12, 5},             // 300 seconds at speeds greater then 25 MPH timer
   {33, 13, 5},             // 600 seconds total drive timer

   {20 + 0*SPACE, 15, 5},   // ECU ID 1
   {20 + 1*SPACE, 15, 5},   // ECU ID 2
   {20 + 2*SPACE, 15, 5},   // ECU ID 3
   {20 + 3*SPACE, 15, 5},   // ECU ID 4
   {20 + 4*SPACE, 15, 5},   // ECU ID 5
   {20 + 5*SPACE, 15, 5},   // ECU ID 6
   {20 + 6*SPACE, 15, 5},   // ECU ID 7
   {20 + 7*SPACE, 15, 5},   // ECU ID 8

   {20 + 0*SPACE, 17, 5},   // initial obdcond, ECU 1
   {20 + 1*SPACE, 17, 5},   // initial obdcond, ECU 2
   {20 + 2*SPACE, 17, 5},   // initial obdcond, ECU 3
   {20 + 3*SPACE, 17, 5},   // initial obdcond, ECU 4
   {20 + 4*SPACE, 17, 5},   // initial obdcond, ECU 5
   {20 + 5*SPACE, 17, 5},   // initial obdcond, ECU 6
   {20 + 6*SPACE, 17, 5},   // initial obdcond, ECU 7
   {20 + 7*SPACE, 17, 5},   // initial obdcond, ECU 8

   {20 + 0*SPACE, 18, 5},   // current obdcond, ECU 1
   {20 + 1*SPACE, 18, 5},   // current obdcond, ECU 2
   {20 + 2*SPACE, 18, 5},   // current obdcond, ECU 3
   {20 + 3*SPACE, 18, 5},   // current obdcond, ECU 4
   {20 + 4*SPACE, 18, 5},   // current obdcond, ECU 5
   {20 + 5*SPACE, 18, 5},   // current obdcond, ECU 6
   {20 + 6*SPACE, 18, 5},   // current obdcond, ECU 7
   {20 + 7*SPACE, 18, 5},   // current obdcond, ECU 8

   {20 + 0*SPACE, 20, 5},   // initial ignctr, ECU 1
   {20 + 1*SPACE, 20, 5},   // initial ignctr, ECU 2
   {20 + 2*SPACE, 20, 5},   // initial ignctr, ECU 3
   {20 + 3*SPACE, 20, 5},   // initial ignctr, ECU 4
   {20 + 4*SPACE, 20, 5},   // initial ignctr, ECU 5
   {20 + 5*SPACE, 20, 5},   // initial ignctr, ECU 6
   {20 + 6*SPACE, 20, 5},   // initial ignctr, ECU 7
   {20 + 7*SPACE, 20, 5},   // initial ignctr, ECU 8

   {20 + 0*SPACE, 21, 5},   // current ignctr, ECU 1
   {20 + 1*SPACE, 21, 5},   // current ignctr, ECU 2
   {20 + 2*SPACE, 21, 5},   // current ignctr, ECU 3
   {20 + 3*SPACE, 21, 5},   // current ignctr, ECU 4
   {20 + 4*SPACE, 21, 5},   // current ignctr, ECU 5
   {20 + 5*SPACE, 21, 5},   // current ignctr, ECU 6
   {20 + 6*SPACE, 21, 5},   // current ignctr, ECU 7
   {20 + 7*SPACE, 21, 5},   // current ignctr, ECU 8

   {60, 11, 4},             // RPM label
   {67, 11, 5},             // RPM value
   {67, 12, 5},             // Speed
};

const int _num_dynamic_elements10 = sizeof(_dynamic_elements10)/sizeof(_dynamic_elements10[0]);

#define IDLE_TIMER          0
#define SPEED_25_MPH_TIMER  1
#define TOTAL_DRIVE_TIMER   2
#define ECU_ID              3
#define INITIAL_OBDCOND     11
#define CURRENT_OBDCOND     19
#define INITIAL_IGNCTR      27
#define CURRENT_IGNCTR      35

#define RPM_LABEL_INDEX     43
#define RPM_INDEX           44
#define SPEED_INDEX         45

#define SetFieldDec(index,val)  (update_screen_dec(_dynamic_elements10,_num_dynamic_elements10,index,val))
#define SetFieldHex(index,val)  (update_screen_hex(_dynamic_elements10,_num_dynamic_elements10,index,val))
#define SetFieldText(index,val) (update_screen_text(_dynamic_elements10,_num_dynamic_elements10,index,val))

/*
*******************************************************************************
** TestToVerifyInUseCounters - Function to run test to verify in-use counters with no faults
*******************************************************************************
*/
STATUS TestToVerifyInUseCounters (void)
{
    int           nNextTest = 0;
    unsigned int  EcuIndex;
    SID_REQ       SidReq;
    STATUS        ret_code;
    unsigned long tEngineStartTimeStamp;
    unsigned long error_count;

    BOOL          bTestFailed = FALSE;

    /* Initialize Array */
    memset (Test10_9_Sid9Inf8, 0x00, sizeof(Test10_9_Sid9Inf8));

    
    /* Prompt user to perform drive cycle to clear I/M readiness bits */
    LogPrint("\n\n**** Test 10.1 (Verify in-use counters) ****\n\n");

    LogUserPrompt("Turn key on without cranking or starting engine.\n"
                  "Press enter to continue.", ENTER_PROMPT);

    /* Engine should now be not running */
    gOBDEngineRunning = FALSE;

    /* Determine the OBD protocol to use */
    gOBDProtocolOrder = 0;
    if (DetermineProtocol() != PASS)
    {
        LogPrint("**** Test 10.1 FAILED ****\n");
        if (LogUserPrompt("Protocol determination failed. Continue?", YES_NO_PROMPT) == 'N')
        {
            return(FAIL);
        }
        gOBDFailureBypassed = TRUE;
    }
    else
    {
        if (gOBDFailureBypassed == FALSE)
        {
            LogPrint("**** Test 10.1 PASSED ****\n");
        }
    }

    /* Get VIN and create (or append) log file */
    LogPrint("\n\n**** Test 10.2 ****\n");

    if ( (nNextTest = StartLogFile()) == 0)
    {
        LogPrint("**** Test 10.2 FAILED ****\n");
        return(FAIL);
    }

    /* Add CALIDs to the log file */
    if (PrintCALIDs () != PASS)
    {
        LogPrint ("FAILURE: Unable to print CALIDs\n");
        ERROR_RETURN;
    }

    LogPrint("**** Test 10.2 PASSED ****\n");

    if (nNextTest == 11)
    {
        return(PASS);
    }

    /* Identify ECUs that support SID 1 PID 1 */
    LogPrint("\n\n**** Test 10.3 ****\n");
    if (RequestSID1SupportData() != PASS)
    {
        LogPrint("**** Test 10.3 FAILED ****\n");
        if (LogUserPrompt("Diagnostic support/data failed. Continue?", YES_NO_PROMPT) == 'N')
        {
            return(FAIL);
        }
        gOBDFailureBypassed = TRUE;
    }
    else
    {
        if (gOBDFailureBypassed == FALSE)
        {
            LogPrint("**** Test 10.3 PASSED ****\n");
        }
    }

    /* Clear DTCs (service 4) */
    LogPrint("\n\n**** Test 10.4 ****\n");
    if (ClearCodes() != PASS)
    {
        LogPrint("**** Test 10.4 FAILED ****\n");
        if (LogUserPrompt("Clear codes failed. Continue?", YES_NO_PROMPT) == 'N')
        {
            return(FAIL);
        }
        gOBDFailureBypassed = TRUE;
    }
    else
    {
        if (gOBDFailureBypassed == FALSE)
        {
            LogPrint("**** Test 10.4 PASSED ****\n");
        }
    }

    gOBDDTCHistorical = FALSE;

    /* Verify I/M readiness is "not ready" (service 1) */
    LogPrint("\n\n**** Test 10.5 ****\n");

    if (VerifyIM_Ready() != PASS)
    {
        LogPrint("**** Test 10.5 FAILED ****\n");
        if (LogUserPrompt("Vehicle information data failed. Continue?", YES_NO_PROMPT) == 'N')
        {
            return(FAIL);
        }
        gOBDFailureBypassed = TRUE;
    }
    else
    {
        if (gOBDFailureBypassed == FALSE)
        {
            LogPrint("**** Test 10.5 PASSED ****\n");
        }
    }

    /* Verify Monitor Resets (service 6) */
    LogPrint("\n\n**** Test 10.6 ****\n");
    if (VerifyMonitorTestSupportAndResults() != PASS)
    {
        LogPrint("**** Test 10.6 FAILED ****\n");
        if (LogUserPrompt("Diagnostic support/data failed. Continue?", YES_NO_PROMPT) == 'N')
        {
            return(FAIL);
        }
        gOBDFailureBypassed = TRUE;
    }
    else
    {
        if (gOBDFailureBypassed == FALSE)
        {
            LogPrint("**** Test 10.6 PASSED ****\n");
        }
    }

    /* Verify no pending DTCs (service 7) */
    LogPrint("\n\n**** Test 10.7 ****\n");
    if (VerifyDTCPendingData() != PASS)
    {
        LogPrint("**** Test 10.7 FAILED ****\n");
        if (LogUserPrompt("Verify DTC Pending Data failed. Continue?", YES_NO_PROMPT) == 'N')
        {
            return(FAIL);
        }
        gOBDFailureBypassed = TRUE;
    }
    else
    {
        if (gOBDFailureBypassed == FALSE)
        {
            LogPrint("**** Test 10.7 PASSED ****\n");
        }
    }

    /* Verify no confirmed DTCs (service 3) */
    LogPrint("\n\n**** Test 10.8 ****\n");
    if (VerifyDTCStoredData() != PASS)
    {
        LogPrint("**** Test 10.8 FAILED ****\n");
        if (LogUserPrompt("Verify DTC Stored Data failed. Continue?", YES_NO_PROMPT) == 'N')
        {
            return(FAIL);
        }
        gOBDFailureBypassed = TRUE;
    }
    else
    {
        if (gOBDFailureBypassed == FALSE)
        {
            LogPrint("**** Test 10.8 PASSED ****\n");
        }
    }

    /* Get in-use performance tracking (service 9 infotype 8) */
    LogPrint("\n\n**** Test 10.9 ****\n");

    /* Sid 9 Inf 8 request*/
    if (IsSid9InfSupported (-1, 0x08) == FALSE)
    {
        LogPrint("FAILURE: Sid 9 Inf $08 not supported\n");
        LogPrint("**** Test 10.9 FAILED ****\n");
        return (FAIL);
    }

    SidReq.SID    = 9;
    SidReq.NumIds = 1;
    SidReq.Ids[0] = 8;

    if ( SidRequest( &SidReq, SID_REQ_NORMAL ) != PASS)
    {
        LogPrint("FAILURE: Sid9 INF request failed\n");
        LogPrint("**** Test 10.9 FAILED ****\n");
        return (FAIL);
    }

    for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
    {
        if (gOBDResponse[EcuIndex].Sid9InfSize > 0)
        {
            if (GetSid9Inf8Data (EcuIndex, &Test10_9_Sid9Inf8[EcuIndex]) != PASS)
            {
                LogPrint("**** Test 10.9 FAILED ****\n");
                return (FAIL);
            }
        }
    }

    if (LogSid9Inf8() != PASS)
    {
        LogPrint("**** Test 10.9 FAILED ****\n");
        return (FAIL);
    }

    LogUserPrompt("Turn ignition off (engine off) for 60 seconds.\n"
                  "Press enter to continue.", ENTER_PROMPT);

    DisconnectProtocol ();

    LogUserPrompt("Turn ignition to crank position and start engine.\n"
                  "Press enter to continue.", ENTER_PROMPT);

    tEngineStartTimeStamp = GetTickCount ();


    LogPrint("**** Test 10.9 PASSED ****\n");

    /* Engine should now be running */
    gOBDEngineRunning = TRUE;

    LogPrint("\n\n**** Test 10.10 ****\n");

    if (ConnectProtocol() != PASS)
    {
        LogPrint("**** Test 10.10 FAILED ****\n");
        if (LogUserPrompt("Protocol determination failed. Continue?", YES_NO_PROMPT) == 'N')
        {
            return(FAIL);
        }
        gOBDFailureBypassed = TRUE;
    }
    else
    {
        if (gOBDFailureBypassed == FALSE)
        {
            LogPrint("**** Test 10.10 PASSED ****\n");
        }
    }
    
    /* User prompt, run test */
    LogPrint("\n\n**** Test 10.11 ****\n");

    StopPeriodicMsg (FALSE);
    Sleep (gOBDRequestDelay);
    LogPrint ("INFORMATION: Stop periodic messages\n");

    gIgnoreNoResponse = TRUE;

    ErrorFlags (ER_BYPASS_USER_PROMPT | ER_CONTINUE);
    ErrorCount();   /* clear error count */

    gSuspendScreenOutput = TRUE;
    ret_code = RunDynamicTest10 (tEngineStartTimeStamp);
    gSuspendScreenOutput = FALSE;

    ErrorFlags (0);

    // re-start tester-present message
    StartPeriodicMsg ();

    gIgnoreNoResponse = FALSE;

    error_count = ErrorCount();

    if (ret_code == PASS && error_count == 0)
    {
        LogPrint("**** Test 10.11 PASSED ****\n");
    }
    else
    {
        bTestFailed = TRUE;

        if (error_count != 0)
        {
            LogPrint ("INFORMATION: Errors detected.\n");
        }

        if (ret_code == ABORT)
        {
            LogPrint("**** Test 10.11 INCOMPLETE ****\n");
        }
        else /* FAIL or PASS w/ errors */
        {
            LogPrint("**** Test 10.11 FAILED ****\n");
        }

        if (LogUserPrompt("Do you wish to continue?\n", YES_NO_PROMPT) == 'N')
        {
            return FAIL;
        }
    }

    LogUserPrompt("Stop the vehicle in a safe location without turning the ignition off.\n\n"
                  "Press enter to continue.", ENTER_PROMPT);

    StopPeriodicMsg (TRUE);

    /* Verify engine warm data (service 1) */
    LogPrint("\n\n**** Test 10.12 ****\n");
    gOBDEngineWarm = TRUE;
    if (VerifyDiagnosticSupportAndData() == FAIL)
    {
        LogPrint("\n\n**** Test 10.12 FAILED ****\n");
        return FAIL;
    }
    
    LogPrint("\n\n**** Test 10.12 PASSED ****\n");

    /* Get in-use performance tracking (service 9 infotype 8) */
    LogPrint("\n\n**** Test 10.13 ****\n");

    /* Sid 9 Inf 8 request*/
    SidReq.SID    = 9;
    SidReq.NumIds = 1;
    SidReq.Ids[0] = 8;

    if ( SidRequest( &SidReq, SID_REQ_NORMAL ) != PASS)
    {
        LogPrint("FAILURE: Sid9 INF request failed\n");
        LogPrint("**** Test 10.13 FAILED ****\n");
        return FAIL;
    }

    if (LogSid9Inf8() != PASS)
    {
        LogPrint("**** Test 10.13 FAILED ****\n");
        return FAIL;
    }

    LogPrint("\n\n**** Test 10.13 PASSED ****\n");

    LogPrint("**** Section 10 of the Dynamic Test has completed %s. ****\n", 
             (bTestFailed != TRUE) ? "successfully" : "unsuccessfully");

    return (bTestFailed != TRUE) ? PASS : FAIL;
}

/*
*******************************************************************************
** substring
*******************************************************************************
*/
char * substring (char * str, const char * substr)
{
    int tok_len = strlen (substr);
    int n       = strlen (str) - tok_len;
    int i;

    for (i=0; i<=n; i++)
    {
        if (str[i] == substr[0])
            if (strncmp (&str[i], substr, tok_len) == 0)
                return &str[i];
    }

    return 0;
}

/*
*******************************************************************************
** StartLogFile - create new log file or open and append to existing one.
**                use VIN as log filename.
*******************************************************************************
*/
int StartLogFile (void)
{
    char buf[256];

    int nNextTest = 0;

    /* Get VIN */
    if (ReadVIN () == PASS)
    {
        /* use VIN as log filename */
        strcpy (gLogFileName, gVIN);
    }
    else
    {
        do
        {
            printf ("\nPROMPT: Unable to obtain VIN\n\nEnter VIN: ");
            scanf ("%s", buf);
        } while ((strlen (buf) + 1) > 18 /*sizeof (gVIN)*/);

        strcpy (gLogFileName, buf);
    }

    /* Check for log file from Tests 5.xx - 9.xx  */
    strcat (gLogFileName, ".log");
    if ( (ghTempLogFile != ghLogFile) && (ghLogFile != NULL) )
    {
        fclose (ghLogFile);
    }

    /* Test if log file already exists */
    ghLogFile = fopen (gLogFileName, "r+");
    if (ghLogFile == NULL)
    {
        /* file does not exist - create file and start with test 10 */
        ghLogFile = fopen (gLogFileName, "w+");
        if (ghLogFile == NULL)
        {
            printf ("FAILURE: Cannot open log file %s\n", gLogFileName);
            return 0;
        }

        /* log file doesn't exist. continue with Test 10.x */
        nNextTest = 10;
    }
    else
    {
        /* scan file for SID9 INF8 data at Test 10.9 */
        ReadSid9Inf8FromLogFile ("**** Test 10.9 ****", "**** Test 10.9", Test10_9_Sid9Inf8);

        /* log file already exists, go to Test 11.x */
        fputs ("\n****************************************************\n", ghLogFile);
        nNextTest = 11;
    }

    /* Application version, build date, OS, etc */
    LogVersionInformation ();

    /* Copy temp log file to actual log file */
    if (AppendLogFile () != PASS)
    {
        printf ("FAILURE: Error copying temp log file to %s\n", gLogFileName);
        return 0;
    }

    /* Echo user responses to the log file */
    LogPrint ("INFORMATION: How many OBD-II ECUs are on this vehicle (1 to 8)?  %d\n", gUserNumEcus);
    LogPrint ("INFORMATION: How many reprogrammable, OBD-II ECUs are on this vehicle (1 to 8)? %d\n", gUserNumEcusReprgm);
    LogPrint ("INFORMATION: Does the vehicle use compression ignition (i.e. diesel)? %s\n", gOBDDieselFlag ? "YES" : "NO");
    LogPrint ("INFORMATION: Is this a hybrid vehicle? %s\n", gOBDHybridFlag ? "YES" : "NO");

    /* done */
    return nNextTest;
}

/*
*******************************************************************************
** LogSid9Inf8
*******************************************************************************
*/
const char szECUID[]    = "ECU ID:";
const char szOBDCOND[]  = "OBDCOND";
const char szIGNCNTR[]  = "IGNCNTR";
const char szCATCOMP1[] = "CATCOMP1";
const char szCATCOND1[] = "CATCOND1";
const char szCATCOMP2[] = "CATCOMP2";
const char szCATCOND2[] = "CATCOND2";
const char szO2COMP1[]  = "O2COMP1";
const char szO2COND1[]  = "O2COND1";
const char szO2COMP2[]  = "O2COMP2";
const char szO2COND2[]  = "O2COND2";
const char szEGRCOMP[]  = "EGRCOMP";
const char szEGRCOND[]  = "EGRCOND";
const char szAIRCOMP[]  = "AIRCOMP";
const char szAIRCOND[]  = "AIRCOND";
const char szEVAPCOMP[] = "EVAPCOMP";
const char szEVAPCOND[] = "EVAPCOND";

STATUS LogSid9Inf8 (void)
{
    unsigned int  EcuIndex;
    SID9INF8      Sid9Inf8;

    for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
    {
        if (gOBDResponse[EcuIndex].Sid9InfSize > 0)
        {
            if (GetSid9Inf8Data (EcuIndex, &Sid9Inf8) != PASS)
            {
                return (FAIL);
            }

            // log OBDCOND, IGNCTR, all OBD monitor and completion counters
            LogPrint("INFORMATION: %s %X\n", szECUID, GetEcuId (EcuIndex) );

            LogPrint("INFORMATION: %-8s = %u\n", szOBDCOND,  Sid9Inf8.OBDCOND);
            LogPrint("INFORMATION: %-8s = %u\n", szIGNCNTR,  Sid9Inf8.IGNCNTR);
            LogPrint("INFORMATION: %-8s = %u\n", szCATCOMP1, Sid9Inf8.CATCOMP1);
            LogPrint("INFORMATION: %-8s = %u\n", szCATCOND1, Sid9Inf8.CATCOND1);
            LogPrint("INFORMATION: %-8s = %u\n", szCATCOMP2, Sid9Inf8.CATCOMP2);
            LogPrint("INFORMATION: %-8s = %u\n", szCATCOND2, Sid9Inf8.CATCOND2);
            LogPrint("INFORMATION: %-8s = %u\n", szO2COMP1,  Sid9Inf8.O2COMP1);
            LogPrint("INFORMATION: %-8s = %u\n", szO2COND1,  Sid9Inf8.O2COND1);
            LogPrint("INFORMATION: %-8s = %u\n", szO2COMP2,  Sid9Inf8.O2COMP2);
            LogPrint("INFORMATION: %-8s = %u\n", szO2COND2,  Sid9Inf8.O2COND2);
            LogPrint("INFORMATION: %-8s = %u\n", szEGRCOMP,  Sid9Inf8.EGRCOMP);
            LogPrint("INFORMATION: %-8s = %u\n", szEGRCOND,  Sid9Inf8.EGRCOND);
            LogPrint("INFORMATION: %-8s = %u\n", szAIRCOMP,  Sid9Inf8.AIRCOMP);
            LogPrint("INFORMATION: %-8s = %u\n", szAIRCOND,  Sid9Inf8.AIRCOND);
            LogPrint("INFORMATION: %-8s = %u\n", szEVAPCOMP, Sid9Inf8.EVAPCOMP);
            LogPrint("INFORMATION: %-8s = %u\n", szEVAPCOND, Sid9Inf8.EVAPCOND);
            LogPrint ("\n");
        }
    }

    return PASS;
}

/*
*******************************************************************************
** ReadSid9Inf8
*******************************************************************************
*/
BOOL ReadSid9Inf8 (const char * szTestSectionEnd, SID9INF8 Sid9Inf8[])
{
    char buf[256];
    char * p;
    int  count;
    unsigned int EcuIndex, EcuId = 0;

    // search for ECU ID
    while (fgets (buf, sizeof(buf), ghLogFile) != 0)
    {
        if (substring (buf, szTestSectionEnd) != 0)     // end of Test XX section
            return FALSE;

        if ( (p = substring (buf, szECUID)) != 0)
        {
            p += sizeof (szECUID);
            EcuId = strtoul (p, NULL, 16);
            break;
        }
    }

    // find EcuIndex
    for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
    {
        if (EcuId == GetEcuId (EcuIndex))
            break;
    }

    if (EcuIndex >= gUserNumEcus)
        return FALSE;

    // read counters
    for (count = 0; count < 16; count++)
    {
        fgets (buf, sizeof(buf), ghLogFile);

        if ( (p = substring (buf, szOBDCOND)) != 0)
        {
            p = substring (p, "=") + 1;            
            Sid9Inf8[EcuIndex].OBDCOND = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szIGNCNTR)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].IGNCNTR = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szCATCOMP1)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].CATCOMP1 = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szCATCOND1)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].CATCOND1 = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szCATCOMP2)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].CATCOMP2 = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szCATCOND2)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].CATCOND2 = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szO2COMP1)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].O2COMP1 = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szO2COND1)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].O2COND1 = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szO2COMP2)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].O2COMP2 = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szO2COND2)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].O2COND2 = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szEGRCOMP)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].EGRCOMP = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szEGRCOND)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].EGRCOND = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szAIRCOMP)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].AIRCOMP = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szAIRCOND)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].AIRCOND = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szEVAPCOMP)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].EVAPCOMP = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        if ( (p = substring (buf, szEVAPCOND)) != 0)
        {
            p = substring (p, "=") + 1;
            Sid9Inf8[EcuIndex].EVAPCOND = (unsigned short)strtoul (p, NULL, 10);
            continue;
        }

        return FALSE;
    }

    // Sid 9 Inf 8 data valid
    Sid9Inf8[EcuIndex].Flags = 1;

    return TRUE;
}

/*
*******************************************************************************
** ReadSid9Inf8FromLogFile
*******************************************************************************
*/
BOOL ReadSid9Inf8FromLogFile (const char * szTestSectionStart, const char * szTestSectionEnd, SID9INF8 Sid9Inf8[])
{
    char buf[256];

    // log file should be open
    if (ghLogFile == NULL)
    {
        printf ("FAILURE: Log File not open\n");
        return FALSE;
    }

    // search from beginning of file
    fseek (ghLogFile, 0, SEEK_SET);

    while (fgets (buf, sizeof(buf), ghLogFile) != 0)
    {
        if (substring (buf, szTestSectionStart) != 0)
        {
            unsigned int EcuIndex;
            for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
            {
                if (ReadSid9Inf8 (szTestSectionEnd, Sid9Inf8) == FALSE)
                    break;
            }
            fseek (ghLogFile, 0, SEEK_END);
            return TRUE;
        }
    }

    // move to end of file
    fseek (ghLogFile, 0, SEEK_END);
    return FALSE;
}

/*
*******************************************************************************
** RunDynamicTest10
*******************************************************************************
*/
STATUS RunDynamicTest10 (unsigned long tEngineStartTimeStamp)
{
    unsigned int   EcuIndex, TestState, bLoop, bFail, bRunTimeSupport;
    unsigned long  t1SecTimer, tDelayTimeStamp;

    unsigned short tTestStartTime, tTestCompleteTime;
    unsigned short tTempTime;
    unsigned short tIdleTime;       // time at idle (stops at 30)
    unsigned short tAtSpeedTime;    // time at speeds > 25mph (stops at 300)
    unsigned short RPM, Speed, RunTime, OBDCond, IgnCnt;
    unsigned short tObdCondTimestamp[OBD_MAX_ECUS];

    SID_REQ        SidReq;
    SID9INF8       Sid9Inf8[OBD_MAX_ECUS];
    SID1          *pSid1;

    // determine PID support
    bRunTimeSupport = IsSid1PidSupported (-1, 0x1F);

    if ((IsSid1PidSupported (-1, 0x0C) == FALSE) && (bRunTimeSupport == FALSE))
    {
        LogPrint("FAILURE: Sid 1 Pids $0C (RPM) and $1F (RUNTM) not supported\n");
        return (FAIL);
    }

    if (IsSid9InfSupported (-1, 0x08) == FALSE)
    {
        LogPrint("FAILURE: Sid 9 Inf $08 not supported\n");
        return (FAIL);
    }

    // initialize static text elements
    init_screen (_string_elements10, _num_string_elements10);

    for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
    {
        // initialize ECU IDs
        SetFieldHex (ECU_ID+EcuIndex, GetEcuId (EcuIndex));

        // initialize initial OBDCONDs
        SetFieldDec (INITIAL_OBDCOND+EcuIndex, Test10_9_Sid9Inf8[EcuIndex].OBDCOND);

        // initialize initial IGNCTRs
        SetFieldDec (INITIAL_IGNCTR+EcuIndex, Test10_9_Sid9Inf8[EcuIndex].IGNCNTR);

        tObdCondTimestamp[EcuIndex] = 0;
    }

    if (bRunTimeSupport == FALSE)
        SetFieldText(RPM_LABEL_INDEX, "RPM:");

    // flush the STDIN stream of any user input before loop
    clear_keyboard_buffer ();

    RPM = Speed = RunTime = OBDCond = IgnCnt = 0;
    tAtSpeedTime = 0;
    tIdleTime = 0;
    tTempTime = 0;
    tTestCompleteTime = 0;
    tTestStartTime = (unsigned short)(tEngineStartTimeStamp / 1000);
    tDelayTimeStamp = t1SecTimer = GetTickCount ();

    TestState = 0;

    //-------------------------------------------
    // loop until test completes
    //-------------------------------------------
    for (;;)
    {
        //-------------------------------------------
        // request RPM - SID 1 PID $0C
        //-------------------------------------------
        if (bRunTimeSupport == FALSE)
        {
            SidReq.SID    = 1;
            SidReq.NumIds = 1;
            SidReq.Ids[0] = 0x0c;

            if (SidRequest( &SidReq, SID_REQ_NO_PERIODIC_DISABLE ) != PASS)
            {
                LogPrint("FAILURE: Sid 1 Pid $0C request failed\n");
                return (FAIL);
            }

            for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
            {
                if (gOBDResponse[EcuIndex].Sid1PidSize > 0)
                {
                    pSid1 = (SID1 *)&gOBDResponse[EcuIndex].Sid1Pid[0];

                    if (pSid1->PID == 0x0c)
                    {
                        RPM = pSid1->Data[0];
                        RPM = RPM << 8 | pSid1->Data[1];

                        // convert from 1 cnt = 1/4 RPM to 1 cnt = 1 RPM
                        RPM >>= 2;
                        break;
                    }
                }
            }

            if (EcuIndex >= gUserNumEcus)
            {
                LogPrint ("FAILURE: Sid 1 Pid $0C missing response\n");
                return (FAIL);
            }
        }

        //-------------------------------------------
        // request Speed - SID 1 PID $0D
        //-------------------------------------------
        SidReq.SID    = 1;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = 0x0d;

        if (SidRequest( &SidReq, SID_REQ_NO_PERIODIC_DISABLE ) != PASS)
        {
            LogPrint("FAILURE: Sid 1 Pid $0D request failed\n");
            return (FAIL);
        }

        for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
        {
            if (gOBDResponse[EcuIndex].Sid1PidSize > 0)
            {
                pSid1 = (SID1 *)&gOBDResponse[EcuIndex].Sid1Pid[0];

                if (pSid1->PID == 0x0d)
                {
                    unsigned long temp = pSid1->Data[0];

                    // convert from km/hr to mile/hr
                    Speed = (unsigned short)( (temp * 6214) / 10000);
                    break;
                }
            }
        }

        if (EcuIndex >= gUserNumEcus)
        {
            LogPrint ("FAILURE: Sid 1 Pid $0D missing response\n");
            return (FAIL);
        }

        //-------------------------------------------
        // request engine RunTime - SID 1 PID $1F
        //-------------------------------------------
        if (bRunTimeSupport == TRUE)
        {
            SidReq.SID    = 1;
            SidReq.NumIds = 1;
            SidReq.Ids[0] = 0x1f;

            if (SidRequest( &SidReq, SID_REQ_NO_PERIODIC_DISABLE ) != PASS)
            {
                LogPrint("FAILURE: Sid 1 Pid $1F request failed\n");
                return (FAIL);
            }

            for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
            {
                if (gOBDResponse[EcuIndex].Sid1PidSize > 0)
                {
                    pSid1 = (SID1 *)&gOBDResponse[EcuIndex].Sid1Pid[0];

                    if (pSid1->PID == 0x1f)
                    {
                        RunTime = pSid1->Data[0];
                        RunTime = RunTime << 8 | pSid1->Data[1];    // 1 cnt = 1 sec
                        break;
                    }
                }
            }

            if (EcuIndex >= gUserNumEcus)
            {
                LogPrint ("FAILURE: Sid 1 Pid $1F missing response\n");
                return (FAIL);
            }
        }
        else
        {
            RunTime = (unsigned short)(GetTickCount () / 1000) - tTestStartTime;
        }

        //-------------------------------------------
        // Get SID 9, INF 8
        //-------------------------------------------
        SidReq.SID    = 9;
        SidReq.NumIds = 1;
        SidReq.Ids[0] = 8;

        if ( SidRequest( &SidReq, SID_REQ_NO_PERIODIC_DISABLE ) != PASS)
        {
            LogPrint("FAILURE: Sid 9 INF 8 request failed\n");
            return FAIL;
        }

        for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
        {
            if (GetSid9Inf8Data (EcuIndex, &Sid9Inf8[EcuIndex]) == PASS)
            {
                SetFieldDec (CURRENT_OBDCOND+EcuIndex, Sid9Inf8[EcuIndex].OBDCOND);
                SetFieldDec (CURRENT_IGNCTR+EcuIndex,  Sid9Inf8[EcuIndex].IGNCNTR);

                // check when OBDCOND counters increment
                if ( (tObdCondTimestamp[EcuIndex] == 0) && 
                     (Sid9Inf8[EcuIndex].OBDCOND > Test10_9_Sid9Inf8[EcuIndex].OBDCOND) )
                {
                    tObdCondTimestamp[EcuIndex] = RunTime;
                }
            }
        }

        //-------------------------------------------
        // Update current PIDs, total engine time
        //-------------------------------------------
        SetFieldDec (SPEED_INDEX, Speed);
        SetFieldDec (TOTAL_DRIVE_TIMER, RunTime);
        if (bRunTimeSupport == FALSE)
            SetFieldDec (RPM_INDEX, RPM);

        //-------------------------------------------
        // Determine phase of dynamic test
        // update screen
        //-------------------------------------------
        bLoop = TRUE;
        while (bLoop)
        {
            bLoop = FALSE;
            switch (TestState)
            {
                case 0:     // wait until idle  (RunTime > 0 or RPM > 450)
                    if ( (bRunTimeSupport == TRUE) ? (RunTime > 0) : (RPM > 450) )
                    {
                        if ( (Speed <= 1) && (tIdleTime < 30) )
                        {
                           TestState = 1;
                           tIdleTime = 0;
                           bLoop = TRUE;
                        }
                        else if ( (Speed >= 25) && (tAtSpeedTime < 300) )
                        {
                           TestState = 2;
                           bLoop = TRUE;
                        }
                        else if ( (tIdleTime >= 30) && (tAtSpeedTime >= 300) )
                        {
                           TestState = 4;
                           bLoop = TRUE;
                        }
                          
                        tTempTime = RunTime;
                    }
                    break;

                case 1:     // 30 seconds continuous time
                    if ((Speed <= 1) && ( (bRunTimeSupport == FALSE) ? (RPM > 450) : 1) )
                    {
                       tIdleTime = min(tIdleTime + RunTime - tTempTime, 30);

                       SetFieldDec (IDLE_TIMER, tIdleTime);

                       if (tIdleTime >= 30)
                       {
                           TestState = 0;
                           bLoop = TRUE;
                       }
                    }
                    else
                    {
                        TestState = 0;
                        bLoop = TRUE;
                    }
                    tTempTime = RunTime;
                    break;
        /*
        **  Note: the Speed check in case 2 and 3 must match.
        **        both must be either "Speed > 25" or "Speed >= 25"
        */
                case 2:     // 300 seconds cumulative time at Speed >= 25 MPH
                    if (Speed >= 25 && ( (bRunTimeSupport == FALSE) ? (RPM > 450) : 1) )
                    {
                        tAtSpeedTime = min(tAtSpeedTime + RunTime - tTempTime, 300);
                        tTempTime = RunTime;

                        SetFieldDec (SPEED_25_MPH_TIMER, tAtSpeedTime);

                        if (tAtSpeedTime >= 300)
                        {
                            TestState = 0;
                            bLoop = TRUE;
                        }
                    }
                    else
                    {
                        TestState = 0;
                        bLoop = TRUE;
                    }
                    break;

                case 4:     // 600 seconds cumulative time
                    if (RunTime >= 600 && ( (bRunTimeSupport == FALSE) ? (RPM > 450) : 1) )
                    {
                        tTestCompleteTime = RunTime;
                        TestState = 5;
                        bLoop = TRUE;
                    }
                    break;

                case 5:     // check for test pass/fail
                    // check if any OBDCOND counters increment too soon or too late
                    bFail = FALSE;
                    for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
                    {
                        if ( (Test10_9_Sid9Inf8[EcuIndex].Flags != 0) && (tObdCondTimestamp[EcuIndex] != 0) )
                        {
                            if (tObdCondTimestamp[EcuIndex] < (tTestCompleteTime - 20) )
                            {
                                LogPrint ("FAILURE: ECU: %X: OBDCOND incremented too soon (RUNTM = %u)\n", 
                                    GetEcuId(EcuIndex), tObdCondTimestamp[EcuIndex]);
                                bFail = TRUE;
                            }

                            if (tObdCondTimestamp[EcuIndex] > (tTestCompleteTime + 20) )
                            {
                                LogPrint ("FAILURE: ECU: %X: OBDCOND incremented too late (RUNTM = %u)\n", 
                                    GetEcuId(EcuIndex), tObdCondTimestamp[EcuIndex]);
                                bFail = TRUE;
                            }
                        }
                    }

                    if (bFail == TRUE)
                    {
                        LogPrint("INFORMATION: Idle Time = %d;  Speed Time = %d;  Run Time = %d\n",
                                 tIdleTime, tAtSpeedTime, RunTime);
                        return FAIL;
                    }

                    // check for OBDCOND for increment
                    for (EcuIndex = 0; EcuIndex < gUserNumEcus; EcuIndex++)
                    {
                        if ( (Test10_9_Sid9Inf8[EcuIndex].Flags != 0) && (tObdCondTimestamp[EcuIndex] == 0) )
                        {
                            break;      // OBDCOND counter not yet incremented - keep running test
                        }
                    }

                    if (EcuIndex >= gUserNumEcus)
                    {
                        LogPrint("INFORMATION: Idle Time = %d;  Speed Time = %d;  Run Time = %d\n",
                                 tIdleTime, tAtSpeedTime, RunTime);
                        return PASS;  // Test complete
                    }

                    // check for timeout
                    if (RunTime >= (tTestCompleteTime + 20) )
                    {
                        LogPrint ("FAILURE: More than 20 seconds has elapsed since the test completed "
                                  "and not all the ECUs have incremented OBDCOND\n");
                        LogPrint("INFORMATION: Idle Time = %d;  Speed Time = %d;  Run Time = %d\n",
                                 tIdleTime, tAtSpeedTime, RunTime);
                        return FAIL;
                    }
                    break;
            } // end switch (TestState)
        } // end while (bLoop)

        //-------------------------------------------
        // Check for ESC key and sleep
        //-------------------------------------------
        do
        {
            if (_kbhit () != 0)
            {
                if (_getch () == 27)    // ESC key
                {
                    LogPrint("INFORMATION: Test 10 aborted by user\n\n");
                    LogPrint("INFORMATION: Idle Time = %d;  Speed Time = %d;  Run Time = %d\n",
                             tIdleTime, tAtSpeedTime, RunTime);
                    return ABORT;
                }
            }

            tDelayTimeStamp = GetTickCount ();

            Sleep ( min (1000 - (tDelayTimeStamp - t1SecTimer), 50) );

        } while (tDelayTimeStamp - t1SecTimer < 1000);

        t1SecTimer = tDelayTimeStamp;
    }

    LogPrint("FAILURE: Error in RunDynamicTest10()\n\n");
    return FAIL;
}
