/*****************************************************************************
 * THIS PROGRAM IS COPYRIGHT (c) 2007 BY DREW TECHNOLOGIES INC.              *
 * THIS PROGRAM IS PROVIDED 'AS IS' WITHOUT WARRANTY OF ANY KIND, EITHER     *
 * EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED          *
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.       *
 * IN NO EVENT SHALL DREWTECH BE LIABLE FOR DAMAGES, INCLUDING ANY GENERAL,  *
 * SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR    *
 * INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED TO LOSS OF DATA   *
 * OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD     *
 * PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER PROGRAMS),  *
 * EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY    *
 * OF SUCH DAMAGES.                                                          *
 ****************************************************************************/
/*
 * This example program reads the AVIT Analog ports using j2534. 
 *
 *	To compile on the AVIT with TCC:
 *		tcc -o AvitAnalogs -ldavit4 AvitAnalogs.c
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <stdarg.h>

#ifdef WIN32
#include <winsock2.h>
#include <windows.h>
#else
#define FALSE 0
#define TRUE  1
#endif

#define DREWTECHONLY
#include "j2534_v0404boilerplate.c"

/* Function return value definitions */
#define PASS					0
#define FAIL					1

#define MAX_MSGS	5
PASSTHRU_MSG RxMsg[MAX_MSGS];

#define CURRENT_SAMPLE_RATE			1000
#define CURRENT_SAMPLES_PER_READING	1
#define CURRENT_READINGS_PER_MSG	10

/*
********************************************************************************
** Main function
*******************************************************************************
*/
int main(int argc, char **argv)
{
	char error_string[256];
	char firm_ver[80],dll_ver[80],api_ver[80];
    SCONFIG params[4];
    SCONFIG_LIST list;
	int RetVal, SlowMode, AverageMode, i, j, k, c, NumActiveChans;
	unsigned long	DeviceID;
	unsigned long	Anal1ChanID, Anal2ChanID;
	unsigned long	NumMsgs, TotalNumRead;
	unsigned long	SampleRate, SamplesPerResult;
	unsigned long	DeltaTime, OldTime, ExpectedDeltaTime;
	int NumJ2534Devices, SelectedJ2534Index;
	long ADValue;

	SlowMode = AverageMode = FALSE;
	SampleRate = 10000;
	SamplesPerResult = 1;

	for ( i = 1; i < argc; i++ )
	{  
		if (( strncmp(argv[i],"-S",2) == 0 ) || ( strncmp(argv[i],"-s",2) == 0 ))
		{  
			SlowMode = TRUE;
			SampleRate = 100;
			printf("Slow mode, sample rate = 100 samples per second\n");
		}
		if (( strncmp(argv[i],"-A",2) == 0 ) || ( strncmp(argv[i],"-a",2) == 0 ))
		{  
			AverageMode = TRUE;
			SamplesPerResult = 50;
			printf("Average mode, 50 samples per result\n");
		}
	}
	if(!SlowMode && !AverageMode)
	{
		printf("Fast mode 10000 samples per second, no averaging\n");
	}
    /*
    ** Find the J2534 interface and attach to the DLL 
    */
#ifdef WIN32
	NumJ2534Devices = EnumerateJ2534DLLs();
	if(NumJ2534Devices <= 0 )
	{
        printf("\nJ2534 devices not found in the registery. %s",J2534BoilerplateErrorResult);
        printf("\nUse the Install program to install the J2534 DLL.");
        exit(1);
	}

	if(NumJ2534Devices>1)
	{
		printf("\nSelect which J2534 device you want to use 1 - %d:\n",NumJ2534Devices);
		for(i=0;i<NumJ2534Devices;i++)
		{
			printf("%d: Vendor: %s; Device: %s\n",i+1,DeviceList[i].VendorName,DeviceList[i].DeviceName);
		}
		
		/* ask the user to pick one */
        do
		{
	    	printf("Enter device number (1-%d): ", NumJ2534Devices);
    	    SelectedJ2534Index = (getchar() - '1');
		}while((SelectedJ2534Index < 0) || (SelectedJ2534Index > (NumJ2534Devices - 1)));
	}
	else
	{
		SelectedJ2534Index = 0;
	}
	printf("\nUsing the J2534 Device from Vendor: %s\n",DeviceList[SelectedJ2534Index].VendorName);

    if (LoadJ2534DLL(DeviceList[SelectedJ2534Index].LibraryName))
    {
        printf("\nJ2534 DLL interface not found or not loaded, %s",J2534BoilerplateErrorResult);
        printf("\nUse the Install program to install the J2534 DLL.");
        exit(1);
    }
#else
    if (LoadJ2534DLL(NULL))
    {
        printf("\nJ2534 DLL interface not found or not loaded");
        exit(1);
    }
#endif
	RetVal = PassThruOpen(NULL, &DeviceID);
	if (RetVal != STATUS_NOERROR)
	{
		if(RetVal == ERR_DEVICE_NOT_CONNECTED)
		{
			printf("\nError: No AVIT found on the network.");
		}
		else
		{
			printf("\nError %d opening the AVIT", (int)RetVal);
		}
		exit(1);
	}		

	RetVal = PassThruReadVersion(DeviceID,firm_ver,dll_ver,api_ver);
	if (RetVal == STATUS_NOERROR)
	{
		printf("\n");
		printf("firmware version: %s\n",firm_ver);	
		printf("dll version: %s\n",dll_ver);	
		printf("api version: %s\n",api_ver);
	}
	else
	{
		PassThruGetLastError(error_string);
		printf("PassThruReadVersion returned %d!\n(%s)\n",RetVal,error_string);
    	PassThruClose(DeviceID);
		exit(1);
	}

    RetVal = PassThruConnect(DeviceID, ANALOG_IN_1, 0, 0, &Anal1ChanID);
    if(RetVal)
    {
		PassThruGetLastError(error_string);
		printf("PassThruConnect returned %d!\n(%s)\n",RetVal,error_string);
    	PassThruClose(DeviceID);
		exit(1);
    }

    /* Set up the Ananlog Readings */
    params[0].Parameter = SAMPLE_RATE;
 	params[0].Value = SampleRate;
   	params[1].Parameter = SAMPLES_PER_READING;
    params[1].Value = SamplesPerResult;
   	params[2].Parameter = READINGS_PER_MSG;
    params[2].Value = CURRENT_READINGS_PER_MSG;
   	params[3].Parameter = ACTIVE_CHANNELS;
    params[3].Value = 0xff;  // Channels 1 - 8
	NumActiveChans = 8;
    list.NumOfParams = 4;
   	list.ConfigPtr = params;

	printf("Sample rate = %d\n", SampleRate);
	printf("Samples per reading = %d\n", SamplesPerResult);
	printf("Readings per Message = %d\n\n", CURRENT_READINGS_PER_MSG);


    /* Must use ChannelID for SET_CONFIG */
    RetVal = PassThruIoctl(Anal1ChanID, SET_CONFIG, &list, NULL);
    if(RetVal)
    {
		PassThruGetLastError(error_string);
		printf("PassThruIoctl returned %d!\n(%s)\n",RetVal,error_string);
    	PassThruClose(DeviceID);
		exit(1);
    }

	for(c = 0; c < 5; c++)
	{ 
		NumMsgs = 0;
		while(NumMsgs == 0)
		{

			NumMsgs = MAX_MSGS;
			RetVal = PassThruReadMsgs(Anal1ChanID, RxMsg, &NumMsgs, 100);
			if((RetVal != STATUS_NOERROR) && (RetVal != ERR_BUFFER_EMPTY))
			{
				printf("\nPassThruRead returned %d!\n",RetVal);
				PassThruClose(DeviceID);
				exit(1);
			}

			if(NumMsgs > 0)
			{
				printf("\nA1 TimeStamp = %ld\n", RxMsg[0].Timestamp);
				/* print the first 10 results */
		        for(j=0;j<(int)RxMsg[0].DataSize;j+=4*NumActiveChans)
		        {
		   		    for ( i = 0; i < 8; i++)
		           	{
		                ADValue = 	(long)((unsigned long)(RxMsg[0].Data[j+3+(i*4)] << 24) | 	\
									(unsigned long)(RxMsg[0].Data[j+2+(i*4)] << 16) | 			\
									(unsigned long)(RxMsg[0].Data[j+1+(i*4)] << 8) | 			\
									(unsigned long)RxMsg[0].Data[j+(i*4)]);
						printf("C%d=%ld, ", i, ADValue);
					}
					printf("\n");
				}
			}
		}
	}

	PassThruDisconnect(Anal1ChanID);
	PassThruClose(DeviceID);
}
