/*****************************************************************************
 * THIS PROGRAM IS COPYRIGHT (c) 2007 BY DREW TECHNOLOGIES INC.              *
 * THIS PROGRAM IS PROVIDED 'AS IS' WITHOUT WARRANTY OF ANY KIND, EITHER     *
 * EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED          *
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.       *
 * IN NO EVENT SHALL DREWTECH BE LIABLE FOR DAMAGES, INCLUDING ANY GENERAL,  *
 * SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR    *
 * INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED TO LOSS OF DATA   *
 * OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD     *
 * PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER PROGRAMS),  *
 * EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY    *
 * OF SUCH DAMAGES.                                                          *
 ****************************************************************************/
/*
 * This is a simple example program that responds to packets on 
 * multiple protocols and is based on the echodaq example.  
 *
 * To compile on the AVIT using tcc:
 *    tcc -o AVIT_echo -ldavit4 AVIT_echo.c
 */

#include <stdio.h>
#include <getopt.h>
#include <stdlib.h>
#include <time.h>

#ifdef WIN32
#include <stdarg.h>
#include <winsock2.h>
#include <windows.h>
#include <string.h>
#else
#include <stdlib.h>
#include <string.h>
#define arm
#define FALSE 0
#endif

#include "j2534_v0404.h"
#include "j2534_v0404boilerplate.c"

PASSTHRU_MSG Msg;
PASSTHRU_MSG PatMsg;
PASSTHRU_MSG FlowMsg;
unsigned long FilterID, ulDeviceID;

#define INVALID_ID	0xFFFFFFFF

#define NUM_MSGS	15
static PASSTHRU_MSG Msg1[NUM_MSGS];
static int opt_verbose = 0;
static unsigned long CAN2Flags;
//static int NumTxed = 0;

static void usage(void)
{
	printf(
	"This program will open the requested protocols and echo all packets\n"
//	"usage: echodaq [-h][-jJvPV791cCmMwWe][-s{AE|AT|BE|BT}]\n"
	"usage: AVIT_echo [-h][-jJvPV7cCmMwW]\n"
	" -h    print this help\n"
	" -V    Turn on Verbose mode\n"
	" -j    use J1850PWM 41K\n"
	" -J    use J1850PWM 83K\n"
	" -v    use J1850VPW\n"
	" -P    use J1850VPW 41K\n"
	" -7    use J1708\n"
//	" -9    use ISO9141\n"
//	" -1    use ISO14230\n"
//	" -e    use UART Echo Byte\n"
	" -c    use CAN\n"
	" -C    use ISO15765\n"
//	" -s AE use SCI_A_ENGINE\n"
//	" -s AT use SCI_A_TRANS\n"
//	" -s AE use SCI_B_ENGINE\n"
//	" -s BT use SCI_B_TRANS\n"
	" -m    use MEDIUM_CAN (29 bit)\n"
	" -M    use MEDIUM_ISO15765\n"
	" -w    use SWCAN (29 bit)\n"
	" -W    use SWCAN_ISO15765\n"
	);
	exit(0);
}

static
char * channel_name(unsigned long ChannelID)
{
	switch (ChannelID)
	{
		case CAN: return "CAN"; 
		case J1850PWM: return "J1850PWM"; 
		case J1850VPW: return "J1850VPW"; 
		case ISO9141: return "ISO9141"; 
		case ISO14230: return "ISO14230"; 
		case ISO15765: return "ISO15765"; 
		case SCI_A_ENGINE: return "SCI_A_ENGINE"; 
		case SCI_A_TRANS: return "SCI_A_TRANS"; 
		case SCI_B_ENGINE: return "SCI_B_ENGINE"; 
		case SCI_B_TRANS: return "SCI_B_TRANS"; 
		case CAN_PS: return "CAN_PS (29 bit)";
		case ISO15765_PS: return "ISO15765_PS"; 
		case SW_CAN_PS: return "SW_CAN_PS (29 bit)";
		case SW_ISO15765_PS: return "SW_ISO15765_PS";
		case J1708_PS: return "J1708";
        case UART_ECHO_BYTE_PS: return "UART Echo Byte";
		default: break;
	}
	return "?unknown?"; 
}

void
check (unsigned long r, const char *s, int line)
{
	if (r != STATUS_NOERROR)
	{
		char buff[256];

		PassThruGetLastError (&buff[0]);
		printf ("\nOops at line %d in %s: J2534 function returned 0x%lx. Error text was:\n'%s'\n",
		line, s, r, buff);
		if(ulDeviceID != INVALID_ID)
		{
	    	PassThruClose(ulDeviceID);
		}
		exit (99);
	}
}

static void
printmsg (PASSTHRU_MSG * mm, const char *s, int write_flag, int verbose)
{
	char buffer[5000];
	unsigned long x;
	char *b = buffer;
	unsigned long n;

	if(!verbose){return;}

	if (mm == NULL)
	{
		printf ("  %s is NULL", s);
		return;
	}
	n = mm->ExtraDataIndex;
	if (write_flag)
	{
		n = mm->DataSize;
		b += sprintf (b, "\n");
	}
	else
	{
		b += sprintf (b, " EDI=%ld, RxS=0x%08lx TS=%ld\n",
			      mm->ExtraDataIndex, mm->RxStatus,
			      mm->Timestamp);
	}
	b += sprintf (b, "  \\__ ");
	for (x = 0; x < n && x < 24; x++)
	{
		b += sprintf (b, "%02x ", mm->Data[x]);
	}
	for (; x < mm->DataSize && x < 24; x++)
	{
		b += sprintf (b, "[%02x] ", mm->Data[x]);
	}
	printf ("  Rx %s DS=%ld, txfl=0x%08lx%s\n", s, mm->DataSize,
		mm->TxFlags, buffer);
}





int
main (int argc, char **argv)
{
	int i, j, k, UsingMongoose, UsingMCS1;
	signed long ret;
	SCONFIG_LIST sconfig_list;
	unsigned long NumMsgs;
	SCONFIG sconfig[10];
	unsigned long opt_J1850, opt_CAN1, opt_CAN2, opt_serial;
	unsigned long chid_J1850, chid_CAN1, chid_CAN2, chid_serial;
	unsigned long ReadDelay_J1850, ReadDelay_CAN1, ReadDelay_CAN2, ReadDelay_serial;
	unsigned long Jbaudrate, Sbaudrate;
	char userinput;
	unsigned long ReadTime, DiffTime, MaxTime; 

	ulDeviceID = INVALID_ID;
	ReadDelay_J1850 = ReadDelay_CAN1 = ReadDelay_CAN2 = ReadDelay_serial = 0; 

	opt_J1850 = opt_CAN1 = opt_CAN2 = opt_serial = 0;
	opt_verbose = 0;
	Jbaudrate = Sbaudrate = 10400;
	/* Command-line options */
	while ((i = getopt(argc, argv, "hjJvP791cCs:mMwWVe")) != EOF)
	{
		switch (i)
		{
			case 'h': usage(); // and exit
			case 'V': opt_verbose = 1; break;
			case 'j': opt_J1850 = J1850PWM; Jbaudrate = 41666; break;
			case 'J': opt_J1850 = J1850PWM; Jbaudrate = 83333; break;
			case 'v': opt_J1850 = J1850VPW; Jbaudrate = 10400; break;
			case 'P': opt_J1850 = J1850VPW; Jbaudrate = 41666; break;
			case '7': opt_serial = J1708_PS; Sbaudrate = 9600;  break;
			case '9': opt_serial = ISO9141; Sbaudrate = 10400;  break;
			case '1': opt_serial = ISO14230; Sbaudrate = 10400;  break;
            case 'e': opt_serial = UART_ECHO_BYTE_PS; Sbaudrate = 10400;  break;
			case 'c': opt_CAN1 = CAN; break;
			case 'C': opt_CAN1 = ISO15765; break;
			case 's':
				if(optarg[0] == 'A' && optarg[1] == 'E')
					opt_serial = SCI_A_ENGINE;
				if(optarg[0] == 'A' && optarg[1] == 'T')
					opt_serial = SCI_A_TRANS;
				if(optarg[0] == 'B' && optarg[1] == 'E')
					opt_serial = SCI_B_ENGINE;
				if(optarg[0] == 'B' && optarg[1] == 'T')
					opt_serial = SCI_B_TRANS;
				if(!opt_serial) usage();
				break;
			case 'm': opt_CAN2 = CAN_PS; break;
			case 'M': opt_CAN2 = ISO15765_PS; break;
			case 'w': opt_CAN2 = SW_CAN_PS; break;
			case 'W': opt_CAN2 = SW_ISO15765_PS; break;
		}
	}

	if ((argc - optind) != 0) usage(); // no extra commands
	if (!(opt_J1850|opt_CAN1|opt_CAN2|opt_serial)) usage(); // at least 1 protocol

	UsingMongoose = FALSE;
	UsingMCS1 = FALSE;
#ifdef WIN32
	printf ("CarDAQ Plus 'C' or Mong MFC 'M' or Mong ISO 'I'\n or VPW 'V' or PWM 'P' or TVIT 'T'>");
	userinput = getchar();
	if((userinput == 'V') || (userinput == 'v'))
	{
		if(LoadJ2534DLL ("MONGV432.DLL"))
		{
			printf ("Error loading the Mongoose VPW DLL.\n");
			exit(1);
		}
		printf ("Mongoose VPW DLL loaded\n");
		UsingMongoose = TRUE;
	}
	else if((userinput == 'P') || (userinput == 'p'))
	{
		if(LoadJ2534DLL ("MONGP432.DLL"))
		{
			printf ("Error loading the Mongoose PWM DLL.\n");
			exit(1);
		}
		printf ("Mongoose VPW DLL loaded\n");
		UsingMongoose = TRUE;
	}
	else if((userinput == 'M') || (userinput == 'm'))
	{
		if(LoadJ2534DLL ("MONGM432.DLL"))
		{
			printf ("Error loading the Mongoose MFC DLL.\n");
			exit(1);
		}
		printf ("Mongoose MFC DLL loaded\n");
		UsingMongoose = TRUE;
	}
    else if((userinput == 'I') || (userinput == 'i'))
    {
        if(LoadJ2534DLL ("MONGI432.DLL"))
        {
            printf ("Error loading the Mongoose ISO DLL.\n");
            exit(1);
        }
        printf ("Mongoose ISO DLL loaded\n");
        UsingMongoose = TRUE;
    }
	else if((userinput == 'T') || (userinput == 't'))
	{
		if(LoadJ2534DLL ("MONGT432.DLL"))
		{
			printf ("Error loading the TVIT DLL.\n");
			exit(1);
		}
		printf ("TVIT DLL loaded\n");
		UsingMongoose = TRUE;
	}
	else if((userinput == 'E') || (userinput == 'e'))
	{
		LoadJ2534DLL ("Mcs132.DLL");
		printf ("EEPOD DLL loaded\n");
		UsingMCS1 = TRUE;
	}
	else
#endif
	{
		if(LoadJ2534DLL ("CDPLS432.DLL"))
		{
			printf ("Error loading the CarDAQ-Plus DLL.\n");
			exit(1);
		}
		printf ("CarDAQ Plus DLL loaded\n");
	}
	printf ("Listening on");

/* Connect to each protocol */

	ret = PassThruOpen(NULL, &ulDeviceID);
	check (ret, __FUNCTION__, __LINE__);

	if(opt_J1850)
	{
		printf (" %s", channel_name(opt_J1850));
		ret = PassThruConnect (ulDeviceID, opt_J1850, 0, Jbaudrate, &chid_J1850);
		check (ret, __FUNCTION__, __LINE__);
		/* only set 1 read delay */
		if(!ReadDelay_J1850 && !ReadDelay_CAN1 && !ReadDelay_CAN2 && !ReadDelay_serial)
		{ 
			ReadDelay_J1850 = 1;
		}
		PatMsg.Data[0] = 0;		// ECU address (CarDAQ Plus under test) for reads
		PatMsg.Data[1] = 0;		// ECU address (CarDAQ Plus under test) for reads
		PatMsg.Data[2] = 0;		// ECU address (CarDAQ Plus under test) for reads

		if(opt_J1850 == J1850PWM)
		{
			PatMsg.DataSize = 3;
		}
		else
		{
			PatMsg.DataSize = 1;
		}
		PatMsg.TxFlags = 0;
		PatMsg.ProtocolID = opt_J1850;

		Msg.Data[0] = 0;
		Msg.Data[1] = 0;
		Msg.Data[2] = 0;

		if(opt_J1850 == J1850PWM)
		{
			Msg.DataSize = 3;
		}
		else
		{
			Msg.DataSize = 1;
		}
		Msg.TxFlags = 0;
		Msg.ProtocolID = opt_J1850;
		ret = PassThruStartMsgFilter (opt_J1850, PASS_FILTER, &Msg,
					      &PatMsg, NULL, &FilterID);
		check (ret, __FUNCTION__, __LINE__);
	}
	if (opt_CAN1)
	{
		printf (" %s", channel_name(opt_CAN1));
		ret = PassThruConnect (ulDeviceID, opt_CAN1, 0, 500000, &chid_CAN1);
		check (ret, __FUNCTION__, __LINE__);
		/* only set 1 read delay */
		if(!ReadDelay_J1850 && !ReadDelay_CAN1 && !ReadDelay_CAN2 && !ReadDelay_serial)
		{ 
			ReadDelay_CAN1 = 1;
		}

		if(opt_CAN1 != ISO15765)
		{
			/* set up the pass filter */
			PatMsg.Data[0] = 0;		// ECU address (CarDAQ Plus under test) for reads
			PatMsg.Data[1] = 0;
			PatMsg.Data[2] = 0;
			PatMsg.Data[3] = 0;

			PatMsg.DataSize = 4;
			PatMsg.TxFlags = 0;
			PatMsg.ProtocolID = opt_CAN1;

			Msg.Data[0] = 0;
			Msg.Data[1] = 0;
			Msg.Data[2] = 0;
			Msg.Data[3] = 0;

			Msg.DataSize = 4;
			Msg.TxFlags = 0;
			Msg.ProtocolID = opt_CAN1;
			ret = PassThruStartMsgFilter (chid_CAN1, PASS_FILTER, &Msg,
						      &PatMsg, NULL, &FilterID);
			check (ret, __FUNCTION__, __LINE__);
		}
	}

	if(opt_CAN2)
	{
        unsigned long Baudrate = 125000;

        CAN2Flags = CAN_29BIT_ID;
        printf (" %s", channel_name(opt_CAN2));
        if (opt_CAN2 == SW_CAN_PS)
        { // Setup Single-wire CAN baud rate
            Baudrate = 33300;
        }
        if (opt_CAN2 == SW_ISO15765_PS)
        { // Setup Single-wire CAN baud rate
            Baudrate = 83300;
        }
        if (opt_CAN2 == SW_ISO15765_PS || opt_CAN2 == ISO15765_PS)
        { // Setup Single-wire CAN flags
            CAN2Flags = 0;
        }
		ret = PassThruConnect (ulDeviceID, opt_CAN2, CAN2Flags, Baudrate, &chid_CAN2);
		check (ret, __FUNCTION__, __LINE__);

		/* only set 1 read delay */
		if(!ReadDelay_J1850 && !ReadDelay_CAN1 && !ReadDelay_CAN2 && !ReadDelay_serial)
		{ 
			ReadDelay_CAN2 = 1;
		}
		PatMsg.Data[0] = 0;		// ECU address (CarDAQ Plus under test) for reads
		PatMsg.Data[1] = 0;
		PatMsg.Data[2] = 0;
		PatMsg.Data[3] = 0;

		PatMsg.DataSize = 4;
		PatMsg.TxFlags = CAN2Flags;
		PatMsg.ProtocolID = opt_CAN2;

		Msg.Data[0] = 0;
		Msg.Data[1] = 0;
		Msg.Data[2] = 0;
		Msg.Data[3] = 0;

		Msg.DataSize = 4;
		Msg.TxFlags = CAN2Flags;
		Msg.ProtocolID = opt_CAN2;
		ret = PassThruStartMsgFilter (chid_CAN2, PASS_FILTER, &Msg,
					      &PatMsg, NULL, &FilterID);
        if (opt_CAN2 != SW_ISO15765_PS && opt_CAN2 != ISO15765_PS)
        { // Setup Single-wire CAN baud rate
            check (ret, __FUNCTION__, __LINE__);
        }
	}

	if(opt_serial)
	{
		printf (" %s", channel_name(opt_serial));
		ret = PassThruConnect (ulDeviceID, opt_serial, 0, Sbaudrate, &chid_serial);
		check (ret, __FUNCTION__, __LINE__);

		/* only set 1 read delay */
		if(!ReadDelay_J1850 && !ReadDelay_CAN1 && !ReadDelay_CAN2 && !ReadDelay_serial)
		{ 
			ReadDelay_serial = 1;
		}
		PatMsg.Data[0] = 0;		// ECU address (CarDAQ Plus under test) for reads
		PatMsg.Data[1] = 0;		// ECU address (CarDAQ Plus under test) for reads

		PatMsg.DataSize = 1;
		PatMsg.TxFlags = 0;
		PatMsg.ProtocolID = opt_serial;

		Msg.Data[0] = 0;
		Msg.Data[1] = 0;

		Msg.DataSize = 1;
		Msg.TxFlags = 0;
		Msg.ProtocolID = opt_serial;
		ret = PassThruStartMsgFilter (chid_serial, PASS_FILTER, &Msg,
					      &PatMsg, NULL, &FilterID);
		check (ret, __FUNCTION__, __LINE__);
	}

	/* Check the analogs, if pulled up, then use high speed mode */
	/* Read the Inputs */
	if( (opt_J1850 == J1850PWM) || (opt_J1850 == J1850VPW) )
	{
		unsigned long ulBatVolts;
		unsigned long ulOnSwitch = 0;

		ret = PassThruIoctl(chid_J1850, READ_VBATT, NULL, &ulBatVolts);

		if(ret == STATUS_NOERROR)
		{
			ret = PassThruIoctl(chid_J1850, READ_CH1_VOLTAGE, NULL, &ulOnSwitch);

			/* if we are in external input mode, update the values */
			if((ulOnSwitch >= (ulBatVolts / 2)) && (ret == STATUS_NOERROR))
			{
				if(opt_J1850 == J1850PWM)
				{
					Jbaudrate = 83333;
				}
				else
				{
					Jbaudrate = 41666;
				}		
				printf("\nAnalog Input says use High speed");
			}
		}
	}
	printf("\n");

/* Protocol-specific initalizations */

	if(opt_J1850)
	{ // Set PWM node address
		if(opt_J1850 == J1850PWM) sconfig_list.NumOfParams = 2;
		else sconfig_list.NumOfParams = 1;
		sconfig_list.ConfigPtr = sconfig;
		sconfig[0].Parameter = DATA_RATE;
		sconfig[0].Value = Jbaudrate;
		sconfig[1].Parameter = NODE_ADDRESS;
		sconfig[1].Value = 0xF1;
		ret = PassThruIoctl (chid_J1850, SET_CONFIG, &sconfig_list,
				     NULL);
		check (ret, __FUNCTION__, __LINE__);
	}


	if(opt_serial == SCI_A_ENGINE || opt_serial == SCI_A_TRANS 
		|| opt_serial == SCI_B_ENGINE || opt_serial == SCI_B_TRANS)
	{
		// set SCI baud rate
		sconfig_list.NumOfParams = 1;
		sconfig_list.ConfigPtr = sconfig;
		sconfig[0].Parameter = DATA_RATE;
		sconfig[0].Value = 62500;
		ret = PassThruIoctl (chid_serial, SET_CONFIG,
			     &sconfig_list, NULL);
		check (ret, __FUNCTION__, __LINE__);
	}

	if(opt_serial == J1708_PS)
	{
		// set SCI baud rate
		sconfig_list.NumOfParams = 2;
		sconfig_list.ConfigPtr = sconfig;
		sconfig[0].Parameter = DATA_RATE;
		sconfig[0].Value = 9600;
		sconfig[1].Parameter = J1962_PINS;
		sconfig[1].Value = 0x0C07;
		ret = PassThruIoctl (chid_serial, SET_CONFIG,
			     &sconfig_list, NULL);
		check (ret, __FUNCTION__, __LINE__);
	}

    if(opt_serial == UART_ECHO_BYTE_PS)
    {
        // set SCI baud rate
        sconfig_list.NumOfParams = 11;
        sconfig_list.ConfigPtr = sconfig;
        sconfig[0].Parameter = J1962_PINS;
        sconfig[0].Value = 0x0700;
        sconfig[1].Parameter = UEB_T0_MIN;
        sconfig[1].Value = 300;
        sconfig[2].Parameter = UEB_T1_MAX;
        sconfig[2].Value = 210;
        sconfig[3].Parameter = UEB_T2_MAX;
        sconfig[3].Value = 20;
        sconfig[4].Parameter = UEB_T3_MAX;
        sconfig[4].Value = 20;
        sconfig[5].Parameter = UEB_T4_MIN;
        sconfig[5].Value = 25;
        sconfig[6].Parameter = UEB_T5_MAX;
        sconfig[6].Value = 231;
        sconfig[7].Parameter = UEB_T6_MAX;
        sconfig[7].Value = 50;
        sconfig[8].Parameter = UEB_T7_MIN;
        sconfig[8].Value = 1;
        sconfig[9].Parameter = UEB_T7_MAX;
        sconfig[9].Value = 55;
        sconfig[10].Parameter = UEB_T9_MIN;
        sconfig[10].Value = 1000;
        ret = PassThruIoctl (chid_serial, SET_CONFIG,
                 &sconfig_list, NULL);
        check (ret, __FUNCTION__, __LINE__);
    }

    if(opt_CAN2 == SW_ISO15765_PS)
    {
        // set SCI baud rate
        sconfig_list.NumOfParams = 1;
        sconfig_list.ConfigPtr = sconfig;
        sconfig[0].Parameter = SW_CAN_RES_SWITCH;
        sconfig[0].Value = DISCONNECT_RESISTOR;
        ret = PassThruIoctl (chid_CAN2, SET_CONFIG,
                 &sconfig_list, NULL);
        check (ret, __FUNCTION__, __LINE__);
        ret = PassThruIoctl (chid_CAN2, SW_CAN_HS, NULL, NULL);
        check (ret, __FUNCTION__, __LINE__);
    }


// setup ISO15765 filter
	if(opt_CAN1 == ISO15765)
	{
		int filt;

		/*
		 * Filter(type=BLOCK_FILTER, mask=0x00 0x00 0x07 0xff, pattern=0x00 0x00 0x01 0x03, flow=)
		 */
		printf ("CAN1 filter Flow pattern = 0060 and flow = 0061\n");

		PatMsg.Data[0] = 0;		// ECU address (CarDAQ Plus under test) for reads
		PatMsg.Data[1] = 0;
		PatMsg.Data[2] = 6;
		PatMsg.Data[3] = 0;

		PatMsg.DataSize = 4;
		PatMsg.TxFlags = 0;
		PatMsg.ProtocolID = ISO15765;

		FlowMsg.Data[0] = 0;   // Tester address (our address) for writes
		FlowMsg.Data[1] = 0;
		FlowMsg.Data[2] = 6;
		FlowMsg.Data[3] = 1;

		FlowMsg.DataSize = 4;
		FlowMsg.TxFlags = 0;
		FlowMsg.ProtocolID = ISO15765;

		Msg.Data[0] = 0xff;
		Msg.Data[1] = 0xff;
		Msg.Data[2] = 0xff;
		Msg.Data[3] = 0xff;

		Msg.DataSize = 4;
		Msg.TxFlags = 0;
		Msg.ProtocolID = ISO15765;

		for(filt=0;filt<4;filt++)
		{
			PatMsg.Data[3] = (unsigned char)(0 + (filt * 2));
			FlowMsg.Data[3] = (unsigned char)(1 + (filt * 2));

			ret = PassThruStartMsgFilter (chid_CAN1, FLOW_CONTROL_FILTER, &Msg,
					      &PatMsg, &FlowMsg, &FilterID);
			check (ret, __FUNCTION__, __LINE__);
		}
	}
// setup ISO15765 filter
	if(opt_CAN2 == SW_ISO15765_PS || opt_CAN2 == ISO15765_PS)
	{
		int filt;

		/*
		 * Filter(type=BLOCK_FILTER, mask=0x00 0x00 0x07 0xff, pattern=0x00 0x00 0x01 0x03, flow=)
		 */
		printf ("CAN2 filter Flow pattern = 0068 and flow = 0069\n");

		PatMsg.Data[0] = 0;		// ECU address (CarDAQ Plus under test) for reads
		PatMsg.Data[1] = 0;
		PatMsg.Data[2] = 6;
		PatMsg.Data[3] = 0;

		PatMsg.DataSize = 4;
		PatMsg.TxFlags = 0;
		PatMsg.ProtocolID = opt_CAN2;

		FlowMsg.Data[0] = 0;   // Tester address (our address) for writes
		FlowMsg.Data[1] = 0;
		FlowMsg.Data[2] = 6;
		FlowMsg.Data[3] = 1;

		FlowMsg.DataSize = 4;
		FlowMsg.TxFlags = 0;
		FlowMsg.ProtocolID = opt_CAN2;

		Msg.Data[0] = 0xff;
		Msg.Data[1] = 0xff;
		Msg.Data[2] = 0xff;
		Msg.Data[3] = 0xff;

		Msg.DataSize = 4;
		Msg.TxFlags = 0;
		Msg.ProtocolID = opt_CAN2;

		for(filt=0;filt<4;filt++)
		{
			PatMsg.Data[3] = (unsigned char)(8 + (filt * 2));
			FlowMsg.Data[3] = (unsigned char)(9 + (filt * 2));

			ret = PassThruStartMsgFilter (chid_CAN2, FLOW_CONTROL_FILTER, &Msg,
					      &PatMsg, &FlowMsg, &FilterID);
			check (ret, __FUNCTION__, __LINE__);
		}
	}


	MaxTime = 0;
	
	/* Make Mongoose go faster */
	if(UsingMongoose) ReadDelay_CAN1 = ReadDelay_serial = 0;

	do
	{
		if (opt_CAN1)
		{
			// CAN
//		ReadTime = GetTickCount();	
			NumMsgs = NUM_MSGS;
			ret = PassThruReadMsgs(chid_CAN1, &Msg1[0], &NumMsgs, ReadDelay_CAN1);
			if (ret == ERR_TIMEOUT || ret == ERR_BUFFER_EMPTY)
				ret = STATUS_NOERROR; // Pretend there was no error
			check (ret, __FUNCTION__, __LINE__);
//		DiffTime = GetTickCount() - ReadTime;
//		if(MaxTime < DiffTime)
//		{
//			MaxTime = DiffTime;
//			printf("MaxTime = %d, NumMsgs = %d\r", MaxTime, NumMsgs);
//		}

			if (NumMsgs != 0)
			{
//printf("saw a message\n");
				for (i = 0; i < NumMsgs; )
				{
					printmsg(&Msg1[i], "C1 Rx", 0, opt_verbose);
					/* clear the flags */ // WHY ?? TODO
					Msg1[i].RxStatus = 0;
					if(opt_CAN1 == CAN) Msg1[i].TxFlags = 0;
					else Msg1[i].TxFlags = ISO15765_FRAME_PAD;
					/* Increment the Address by 1 */
					Msg1[i].Data[3]++;

					/* get rid of indication messages */
					if (Msg1[i].ExtraDataIndex == 0)
					{
						k = i;
						for(j=i+1;j<NumMsgs; j++)
						{
							Msg1[k++] = Msg1[j];
						}
						NumMsgs -= 1;
					}
					else
					{
						i++;
					}
				}
				
				if(NumMsgs > 0)
				{
					// send messages back
					if(UsingMCS1)
					{
						ret = PassThruWriteMsgs (chid_CAN1, &Msg1[0], &NumMsgs, 100);
					}
					else
					{
						ret = PassThruWriteMsgs (chid_CAN1, &Msg1[0], &NumMsgs, 0);
					}
					if( (ret == ERR_BUFFER_FULL) || (ret == ERR_TIMEOUT) )
						ret = STATUS_NOERROR; // Pretend there was no error
					check(ret, __FUNCTION__, __LINE__);
				}
			}
		}


		if (opt_CAN2)
		{
			// CAN2
			NumMsgs = NUM_MSGS;
			ret = PassThruReadMsgs(chid_CAN2, &Msg1[0], &NumMsgs, ReadDelay_CAN2);
			if (ret == ERR_TIMEOUT || ret == ERR_BUFFER_EMPTY)
				ret = STATUS_NOERROR; // Pretend there was no error
			check (ret, __FUNCTION__, __LINE__);

			if (NumMsgs != 0)
			{
				for (i = 0; i < NumMsgs; )
				{
					printmsg(&Msg1[i], "C2 Rx", 0, opt_verbose);
					/* clear the flags */ // WHY ?? TODO
					Msg1[i].RxStatus = 0;
                    if(opt_CAN2 == SW_CAN_PS || opt_CAN2 == CAN_PS) Msg1[i].TxFlags = CAN2Flags;
                    else Msg1[i].TxFlags = ISO15765_FRAME_PAD;
					/* Increment the Address by 1 */
					Msg1[i].Data[3]++;
                    /* get rid of indication messages */
                    if (Msg1[i].ExtraDataIndex == 0)
                    {
                        k = i;
                        for(j=i+1;j<NumMsgs; j++)
                        {
                            Msg1[k++] = Msg1[j];
                        }
                        NumMsgs -= 1;
                    }
                    else
                    {
                        i++;
                    }
				}
                if(NumMsgs > 0)
                {
                    // send messages back
                    ret = PassThruWriteMsgs (chid_CAN2, &Msg1[0], &NumMsgs, 0);
                    check(ret, __FUNCTION__, __LINE__);
                }
			}
		}



		if (opt_serial)
		{
			NumMsgs = NUM_MSGS;
			ret = PassThruReadMsgs (chid_serial, &Msg1[0], &NumMsgs, ReadDelay_serial);
			if (ret == ERR_TIMEOUT || ret == ERR_BUFFER_EMPTY)
				ret = STATUS_NOERROR; // Pretend there was no error
			check (ret, __FUNCTION__, __LINE__);

			if (NumMsgs != 0)
			{
				for (i = 0; i < NumMsgs; )
				{
					unsigned char temp;
					printmsg(&Msg1[i], "S Rx", 0, opt_verbose);

					/* clear the flags */
					Msg1[i].RxStatus = 0;
					Msg1[i].TxFlags = 0;

					if((opt_serial != J1708_PS) && (opt_serial != UART_ECHO_BYTE_PS))
					{
						/* swap the source and destination */
						temp = Msg1[i].Data[1];
						Msg1[i].Data[1] = Msg1[i].Data[2];
						Msg1[i].Data[2] = temp;
					}
                    if(opt_serial == UART_ECHO_BYTE_PS)
                    {
                        /* Increment the message number */
                        Msg1[i].Data[1]++;
                    }
					if (Msg1[i].ExtraDataIndex == 0)
					{
						if((Msg1[i].RxStatus & START_OF_MESSAGE) != 0)
						{
							printf ("Serial CRC ERROR\n");
							printmsg(&Msg1[i], "Rx", 0, 1);
					   	}
					   	k = i;
					   	for(j=i+1;j<NumMsgs; j++)
					   	{
					   		Msg1[k++] = Msg1[j];
						}
						NumMsgs -= 1;
					}
					else
					{
						if(opt_serial == J1708_PS)
						{
							/* Set Priority */
							Msg1[i].TxFlags = 1 << 16;
							Msg1[i].Data[0] = 0xFF;  // set first byte to worst case byte
						}
						else
						{
							Msg1[i].DataSize = Msg1[i].ExtraDataIndex;
						}
						i++;
					}
				}
//NumTxed += (int)NumMsgs;
//printf("%d\r",NumTxed);
				if(NumMsgs)
				{
					ret = PassThruWriteMsgs (chid_serial, &Msg1[0], &NumMsgs, 0);
					check (ret, __FUNCTION__, __LINE__);
				}
			}
		}


		if(opt_J1850)
		{
			
			NumMsgs = NUM_MSGS;
			ret = PassThruReadMsgs (chid_J1850, &Msg1[0], &NumMsgs, ReadDelay_J1850);
			if (ret == ERR_TIMEOUT || ret == ERR_BUFFER_EMPTY)
				ret = STATUS_NOERROR; // Pretend there was no error
			check (ret, __FUNCTION__, __LINE__);

			if (NumMsgs != 0)
			{
				for (i = 0; i < NumMsgs; )
				{
					unsigned char temp;
					printmsg(&Msg1[i], "J Rx", 0, opt_verbose);

					if (Msg1[i].RxStatus != 0)
					{
						printf ("J1850 RxStatus = 0x%x, i = %d, NumMsgs = %d\n",(int)Msg1[i].RxStatus, i, NumMsgs);
						printmsg(&Msg1[i], "Rx", 0, 1);
		 //				if(i> 0) printmsg(&Msg1[i-1], "Rx", 0, 1);
		 //				if(i> 1) printmsg(&Msg1[i-2], "Rx", 0, 1);
						k = i;
						for(j=i+1;j<NumMsgs; j++)
						{
							Msg1[k++] = Msg1[j];
						}
						NumMsgs -= 1;
					}
					else if (Msg1[i].ExtraDataIndex == 0)
					{
						printf ("J1850 CRC ERROR\n");
						printmsg(&Msg1[i], "Rx", 0, 1);
						k = i;
						for(j=i+1;j<NumMsgs; j++)
						{
							Msg1[k++] = Msg1[j];
						}
						NumMsgs -= 1;
					}
					else
					{
						/* clear the flags */
						Msg1[i].RxStatus = 0;
						Msg1[i].TxFlags = 0;

						/* swap the source and destination */
						temp = Msg1[i].Data[1];
						Msg1[i].Data[1] = Msg1[i].Data[2];
						Msg1[i].Data[2] = temp;
		
						Msg1[i].DataSize = Msg1[i].ExtraDataIndex;
			
						i++;
					}
				}
				if(NumMsgs)
				{
					ret = PassThruWriteMsgs (chid_J1850, &Msg1[0], &NumMsgs, 0);
					check (ret, __FUNCTION__, __LINE__);
				}
			}
		}

	} // while
#ifdef WIN32
	while(!kbhit());
	/* hit a key to exit the program */

	userinput = getch();
#else
	while(1);
#endif
    
    PassThruClose(ulDeviceID);

	return 0;
}// main

